#!/usr/bin/env python3
import aiohttp
import asyncio
import math

from constants import CONST
from objects import Issue, Event
from log import get_logger


async def change_status(client, issue):
    url = CONST.ST_TRANSITION_URL % {'issue_key': issue.key}
    data = {'comment': CONST.COMMENT}
    async with client.post(url, json=data, headers=CONST.HEADERS) as resp:
        if resp.status == 200:
            get_logger().info('Issue %s transited', issue.key)
        else:
            get_logger().error(str(resp))


async def perform_changelog(client, issue):
    url = CONST.ST_CHANGELOG_URL % {'issue_key': issue.key}
    params = {}

    last_wf_date = issue.created_at
    last_action = CONST.MONTH_TS
    last_wf_event = '-'
    last_action_event = '-'

    while True:
        async with client.get(url, params=params, headers=CONST.HEADERS) as resp:
            if resp.status != 200:
                get_logger().error(str(resp))
                return

            data = await resp.json()
            if not data:
                break

            for event in data:
                event = Event(issue, event)
                params['id'] = event.id

                if event.type == 'IssueWorkflow':
                    last_wf_date = event.date
                    last_wf_event = event.id

                if event.author != issue.assignee and event.comment_added or event.comment_updated:
                    last_action = event.date
                    last_action_event = event.id

    if last_wf_date <= CONST.MONTH_TS and last_action <= CONST.MONTH_TS:
        action = 'ready to transit'
        if CONST.ACTIVE:
            await change_status(client, issue)
    else:
        action = 'out of date interval'

    get_logger().info(
        'Issue %s - %s Last wf %s %s; last action %s %s',
        issue.key, action, last_wf_event, last_wf_date.isoformat(), last_action_event, last_action.isoformat()
    )


async def perform_issues(client, page):
    params = {'perPage': str(CONST.PAGE_SIZE), 'page': str(page)}

    request = client.post(
        CONST.ST_ISSUE_SEARCH_URL,
        json=CONST.ST_ISSUES_QUERY,
        params=params,
        headers=CONST.HEADERS
    )
    async with request as resp:
        if resp.status != 200:
            get_logger().error(str(resp))
            return

        result = await resp.json()
        get_logger().info('Page %d loaded', page)
        if not result:
            return

        tasks = []
        for issue in result:
            issue = Issue(issue)
            if issue.author == issue.assignee:
                get_logger().info('Isuue %s - author == assignee', issue.key)
                continue

            tasks.append(perform_changelog(client=client, issue=issue))
        if tasks:
            await asyncio.wait(tasks)


async def perform():
    get_logger().info('MONTH_TS=%s', CONST.MONTH_TS.isoformat())

    async with aiohttp.ClientSession() as client:
        request = client.post(
            CONST.ST_ISSUE_COUNT_URL,
            json=CONST.ST_ISSUES_QUERY,
            headers=CONST.HEADERS,
        )
        async with request as resp:
            count = await resp.json()
            get_logger().info('Found %d issues', count)
            count = math.ceil(count / CONST.PAGE_SIZE)

        if count:
            tasks = [
                perform_issues(client, page)
                for page in range(1, count + 1)
            ]
            await asyncio.wait(tasks)


if __name__ == '__main__':
    get_logger().info('Start')
    loop = asyncio.get_event_loop()
    try:
        loop.run_until_complete(perform())
    except Exception:
        get_logger().exception('Boom!')
    finally:
        loop.run_until_complete(asyncio.sleep(0.250))
        loop.stop()
    get_logger().info('Finish')
