"""
Base settings to build other settings files upon.
"""
import environ

from lms.courses.utils import files_upload_destination

ROOT_DIR = (
    environ.Path(__file__) - 3
)  # (lms/config/settings/base.py - 3 = lms/)
APPS_DIR = ROOT_DIR.path("lms")

env = environ.Env()

READ_DOT_ENV_FILE = env.bool("DJANGO_READ_DOT_ENV_FILE", default=False)
if READ_DOT_ENV_FILE:
    # OS environment variables take precedence over variables from .env
    env.read_env(str(ROOT_DIR.path(".envs", ".local", ".env")))

# GENERAL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#debug
DEBUG = env.bool("DJANGO_DEBUG", False)
# Local time zone. Choices are
# http://en.wikipedia.org/wiki/List_of_tz_zones_by_name
# though not all of them may be available with every OS.
# In Windows, this must be set to your system time zone.
TIME_ZONE = "UTC"
# https://docs.djangoproject.com/en/dev/ref/settings/#language-code
LANGUAGE_CODE = "ru-RU"
# https://docs.djangoproject.com/en/dev/ref/settings/#site-id
SITE_ID = 1
# https://docs.djangoproject.com/en/dev/ref/settings/#use-i18n
USE_I18N = True
# https://docs.djangoproject.com/en/dev/ref/settings/#use-l10n
USE_L10N = True
# https://docs.djangoproject.com/en/dev/ref/settings/#use-tz
USE_TZ = True
# https://docs.djangoproject.com/en/dev/ref/settings/#locale-paths
LOCALE_PATHS = [ROOT_DIR.path("locale")]

SHORT_DATE_FORMAT = 'Y-m-d'

SHORT_DATETIME_FORMAT = 'Y-m-d H:i'

# DATABASES
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#databases
# DATABASES = {"default": env.db("DATABASE_URL")}
# DATABASES["default"]["ATOMIC_REQUESTS"] = True
DATABASES = {
    "default": {
        "ENGINE": 'django_pgaas.backend',
        "NAME": env('POSTGRES_DB', default='lms'),
        "USER": env('POSTGRES_USER', default='lms'),
        "PASSWORD": env('POSTGRES_PASSWORD'),
        "HOST": env('POSTGRES_HOST', default='127.0.0.1'),
        "POST": env.int('POSTGRES_PORT', default=5432),
        "CONN_MAX_AGE": env.int("CONN_MAX_AGE", default=60),
    }
}

# FILE STORAGE
# ------------------------------------------------------------------------------
DEFAULT_FILE_STORAGE = 'storages.backends.s3boto3.S3Boto3Storage'

# S3 default bucket
# ------------------------------------------------------------------------------
AWS_ACCESS_KEY_ID = env('AWS_ACCESS_KEY_ID', default='minio')
AWS_SECRET_ACCESS_KEY = env('AWS_SECRET_ACCESS_KEY', default='password')
AWS_S3_ENDPOINT_URL = env('AWS_S3_ENDPOINT_URL', default='http://minio:9000')
AWS_STORAGE_BUCKET_NAME = env('AWS_STORAGE_BUCKET_NAME', default='lms-media')

# S3 upload settings
# ------------------------------------------------------------------------------
S3UPLOAD_BUCKET_ENDPOINT = env("S3UPLOAD_BUCKET_ENDPOINT", default='http://127.0.0.1:9000/lms-media')
S3UPLOAD_DESTINATIONS = env.json(
    "S3UPLOAD_DESTINATIONS",
    default={
        "files": {
            "key": files_upload_destination,
        }
    }
)

S3_SCORM_BASE_PATH = env('S3_SCORM_BASE_PATH', default='scorms')

# Achievery settings
# ------------------------------------------------------------------------------
ACHIEVERY_API_URL = env('ACHIEVERY_API_URL', default='https://staff.yandex-team.ru/api/achievery')
ACHIEVERY_OAUTH_TOKEN = env('ACHIEVERY_OAUTH_TOKEN', default='')

# URLS
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#root-urlconf
ROOT_URLCONF = "config.urls"
# https://docs.djangoproject.com/en/dev/ref/settings/#wsgi-application
WSGI_APPLICATION = "config.wsgi.application"

# APPS
# ------------------------------------------------------------------------------
DJANGO_APPS = [
    "django.contrib.auth",
    "django.contrib.contenttypes",
    "django.contrib.sessions",
    # "django.contrib.sites",
    "django.contrib.messages",
    "django.contrib.staticfiles",
    # "django.contrib.humanize", # Handy template tags
    "django_admin_env_notice",
    "django.contrib.admin",
    "django.contrib.postgres",
]
THIRD_PARTY_APPS = [
    "rest_framework",
    "drf_spectacular",
    "django_object_actions",
    "django_celery_beat",
    "django_ltree",
    "django_idm_api",
    "guardian",
    "ordered_model",
    "import_export",
    "django_admin_listfilter_dropdown",
    "storages",
    "simple_history",
    "s3upload",
]

LOCAL_APPS = [
    "lms.contrib.transfer_manager",
    "lms.moduletypes",
    "lms.preferences",
    "lms.users",
    "lms.courses",
    "lms.classrooms",
    "lms.resources",
    "lms.enrollments",
    "lms.staff",
    "lms.tracker",
    "lms.idm",
    "lms.stress",
    "lms.mailing",
    "lms.scorm",
    "lms.reports",
    "lms.calendars",
    "lms.assignments",
    "lms.tags",
    "lms.mentorships",
    "lms.actions",
    # Your stuff: custom apps go here
]
# https://docs.djangoproject.com/en/dev/ref/settings/#installed-apps
INSTALLED_APPS = DJANGO_APPS + THIRD_PARTY_APPS + LOCAL_APPS

# MIGRATIONS
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#migration-modules
# MIGRATION_MODULES = {"sites": "lms.contrib.sites.migrations"}

# AUTHENTICATION
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#authentication-backends
AUTHENTICATION_BACKENDS = [
    "django.contrib.auth.backends.ModelBackend",
    "guardian.backends.ObjectPermissionBackend",
]
# https://docs.djangoproject.com/en/dev/ref/settings/#auth-user-model
AUTH_USER_MODEL = "users.User"
# https://docs.djangoproject.com/en/dev/ref/settings/#login-redirect-url
# LOGIN_REDIRECT_URL = "users:redirect"
# https://docs.djangoproject.com/en/dev/ref/settings/#login-url
# LOGIN_URL = "account_login"

# PASSWORDS
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#password-hashers
PASSWORD_HASHERS = [
    # https://docs.djangoproject.com/en/dev/topics/auth/passwords/#using-argon2-with-django
    "django.contrib.auth.hashers.Argon2PasswordHasher",
    "django.contrib.auth.hashers.PBKDF2PasswordHasher",
    "django.contrib.auth.hashers.PBKDF2SHA1PasswordHasher",
    "django.contrib.auth.hashers.BCryptSHA256PasswordHasher",
]
# https://docs.djangoproject.com/en/dev/ref/settings/#auth-password-validators
# AUTH_PASSWORD_VALIDATORS = [
#     {
#         "NAME": "django.contrib.auth.password_validation.UserAttributeSimilarityValidator"
#     },
#     {"NAME": "django.contrib.auth.password_validation.MinimumLengthValidator"},
#     {"NAME": "django.contrib.auth.password_validation.CommonPasswordValidator"},
#     {"NAME": "django.contrib.auth.password_validation.NumericPasswordValidator"},
# ]

# MIDDLEWARE
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#middleware
MIDDLEWARE = [
    "lms.contrib.logger.middleware.LogRequestIdMiddleware",
    "django.middleware.security.SecurityMiddleware",
    "django.contrib.sessions.middleware.SessionMiddleware",
    "django.middleware.locale.LocaleMiddleware",
    "django.middleware.common.CommonMiddleware",
    "django.middleware.csrf.CsrfViewMiddleware",
    # "django.contrib.auth.middleware.AuthenticationMiddleware",
    "django.contrib.messages.middleware.MessageMiddleware",
    "django.middleware.clickjacking.XFrameOptionsMiddleware",
    "simple_history.middleware.HistoryRequestMiddleware",
]

# STATIC
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#static-root
STATIC_ROOT = str(ROOT_DIR("staticfiles"))
# https://docs.djangoproject.com/en/dev/ref/settings/#static-url
STATIC_URL = "/static/"
# https://docs.djangoproject.com/en/dev/ref/contrib/staticfiles/#std:setting-STATICFILES_DIRS
STATICFILES_DIRS = [str(APPS_DIR.path("static"))]
# https://docs.djangoproject.com/en/dev/ref/contrib/staticfiles/#staticfiles-finders
STATICFILES_FINDERS = [
    "django.contrib.staticfiles.finders.FileSystemFinder",
    "django.contrib.staticfiles.finders.AppDirectoriesFinder",
]

# MEDIA
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#media-root
MEDIA_ROOT = str(APPS_DIR("media"))
# https://docs.djangoproject.com/en/dev/ref/settings/#media-url
MEDIA_URL = "/media/"

# TEMPLATES
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#templates
TEMPLATES = [
    {
        # https://docs.djangoproject.com/en/dev/ref/settings/#std:setting-TEMPLATES-BACKEND
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        # https://docs.djangoproject.com/en/dev/ref/settings/#template-dirs
        "DIRS": [str(APPS_DIR.path("templates"))],
        "OPTIONS": {
            # https://docs.djangoproject.com/en/dev/ref/settings/#template-loaders
            # https://docs.djangoproject.com/en/dev/ref/templates/api/#loader-types
            "loaders": [
                "django.template.loaders.filesystem.Loader",
                "django.template.loaders.app_directories.Loader",
            ],
            # https://docs.djangoproject.com/en/dev/ref/settings/#template-context-processors
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.template.context_processors.i18n",
                "django.template.context_processors.media",
                "django.template.context_processors.static",
                "django.template.context_processors.tz",
                "django.contrib.messages.context_processors.messages",
                "django_admin_env_notice.context_processors.from_settings",
            ],
        },
    }
]

# FIXTURES
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#fixture-dirs
FIXTURE_DIRS = (str(APPS_DIR.path("fixtures")),)

# SECURITY
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#session-cookie-httponly
SESSION_COOKIE_HTTPONLY = True
# https://docs.djangoproject.com/en/dev/ref/settings/#csrf-cookie-httponly
CSRF_COOKIE_HTTPONLY = True
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-browser-xss-filter
SECURE_BROWSER_XSS_FILTER = True
# https://docs.djangoproject.com/en/dev/ref/settings/#x-frame-options
X_FRAME_OPTIONS = "DENY"
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-proxy-ssl-header
SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")

# EMAIL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#email-backend
EMAIL_BACKEND = env(
    "DJANGO_EMAIL_BACKEND", default="django.core.mail.backends.smtp.EmailBackend"
)
# https://docs.djangoproject.com/en/2.2/ref/settings/#email-timeout
EMAIL_TIMEOUT = 5

# ADMIN
# ------------------------------------------------------------------------------
# Django Admin URL.
ADMIN_ENABLED = env.bool("DJANGO_ADMIN_ENABLED", default=False)

ADMIN_URL = "admin/"
# https://docs.djangoproject.com/en/dev/ref/settings/#admins
ADMINS = [
    ("""Dmitry Gerasimenko""", "kidig@yandex-team.ru"),
]
# https://docs.djangoproject.com/en/dev/ref/settings/#managers
MANAGERS = ADMINS

# LOGGING
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#logging
# See https://docs.djangoproject.com/en/dev/topics/logging for
# more details on how to customize your logging configuration.
LOG_LEVEL = env('DJANGO_LOG_LEVEL', default='INFO')

LOGGING = {
    "version": 1,
    "disable_existing_loggers": False,
    "formatters": {
        "verbose": {
            "format": "%(levelname)s %(asctime)s %(module)s "
            "%(process)d %(thread)d %(message)s"
        }
    },
    "handlers": {
        "console": {
            "level": "DEBUG",
            "class": "logging.StreamHandler",
            "formatter": "verbose",
        }
    },
    "root": {"level": LOG_LEVEL, "handlers": ["console"]},
}

# django-rest-framework
# ------------------------------------------------------------------------------
REST_FRAMEWORK = {
    'DATETIME_FORMAT': '%Y-%m-%dT%H:%M:%SZ',
    'DATETIME_INPUT_FORMATS': ['%Y-%m-%dT%H:%M:%SZ', '%Y-%m-%dT%H:%M:%S.%fZ'],
    'DATE_FORMAT': '%Y-%m-%d',
    'TIME_FORMAT': '%H:%M',

    'TEST_REQUEST_DEFAULT_FORMAT': 'json',

    'DEFAULT_FILTER_BACKENDS': [
        'django_filters.rest_framework.DjangoFilterBackend',
    ],

    'DEFAULT_PAGINATION_CLASS': 'rest_framework.pagination.LimitOffsetPagination',
    'PAGE_SIZE': 10,

    'DEFAULT_VERSION': 'v1',
    'DEFAULT_VERSIONING_CLASS': 'rest_framework.versioning.URLPathVersioning',

    'DEFAULT_SCHEMA_CLASS': 'lms.core.views.schema.AutoSchema',

    'DEFAULT_AUTHENTICATION_CLASSES': (
        'lms.core.views.authentication.CsrfExemptSessionAuthentication',
        # 'lms.contrib.tvm.authentication.TVMUserTicketAuthentication',
        # 'rest_framework.authentication.SessionAuthentication',
    ),

    'DEFAULT_RENDERER_CLASSES': (
        'djangorestframework_camel_case.render.CamelCaseJSONRenderer',
        # 'djangorestframework_camel_case.render.CamelCaseBrowsableAPIRenderer',
    ),

    'DEFAULT_PARSER_CLASSES': (
        # 'djangorestframework_camel_case.parser.CamelCaseFormParser',
        # 'djangorestframework_camel_case.parser.CamelCaseMultiPartParser',
        'djangorestframework_camel_case.parser.CamelCaseJSONParser',
    ),
}

# drf-spectacular
# ------------------------------------------------------------------------------
SWAGGER_ENABLED = env.bool('DJANGO_SWAGGER_ENABLED', default=False)

SPECTACULAR_SETTINGS = {
    'VERSION': '1.0.0',
    'SCHEMA_PATH_PREFIX': r'/(lab)?api/v[1-9]',
    # 'SERVE_PERMISSIONS': [
    #     'rest_framework.permissions.IsAuthenticated',
    # ]

    'GET_LIB_DOC_EXCLUDES': 'lms.core.views.schema.get_lib_doc_excludes',

    'POSTPROCESSING_HOOKS': [
        'drf_spectacular.hooks.postprocess_schema_enums',
        'drf_spectacular.contrib.djangorestframework_camel_case.camelize_serializer_fields',
        'lms.core.views.schema.postprocess_schema_title',
    ],
}


# Celery
# ------------------------------------------------------------------------------
if USE_TZ:
    # http://docs.celeryproject.org/en/latest/userguide/configuration.html#std:setting-timezone
    CELERY_TIMEZONE = TIME_ZONE
# http://docs.celeryproject.org/en/latest/userguide/configuration.html#std:setting-broker_url
# CELERY_BROKER_URL = env("CELERY_BROKER_URL")
# http://docs.celeryproject.org/en/latest/userguide/configuration.html#std:setting-result_backend
CELERY_ACCEPT_CONTENT = ["json"]
# http://docs.celeryproject.org/en/latest/userguide/configuration.html#std:setting-task_serializer
CELERY_TASK_SERIALIZER = "json"
# http://docs.celeryproject.org/en/latest/userguide/configuration.html#std:setting-result_serializer
CELERY_RESULT_SERIALIZER = "json"
# http://docs.celeryproject.org/en/latest/userguide/configuration.html#task-soft-time-limit
CELERY_TASK_SOFT_TIME_LIMIT = env.int('CELERY_TASK_SOFT_TIME_LIMIT', default=60)
# http://docs.celeryproject.org/en/latest/userguide/configuration.html#task-time-limit
CELERY_TASK_TIME_LIMIT = max(env.int('CELERY_TASK_TIME_LIMIT', default=5 * 60), CELERY_TASK_SOFT_TIME_LIMIT)
# http://docs.celeryproject.org/en/latest/userguide/configuration.html#beat-scheduler
CELERY_BEAT_SCHEDULER = "beatlock.schedulers.BeatLockScheduler"

CELERY_CREATE_MISSING_QUEUES = True

CELERY_TASK_DEFAULT_QUEUE = 'lms_celery'
CELERY_TASK_FAST_DISK_QUEUE = 'lms_celery_fast_disk'

CELERY_ROUTES = {
    'lms.scorm.tasks.*': {'queue': CELERY_TASK_FAST_DISK_QUEUE},
}

CELERY_BATCH_SIZE_DEFAULT = env.int('CELERY_BATCH_SIZE_DEFAULT', default=1_000)

CELERY_LOGGING = {
    "version": 1,
    "disable_existing_loggers": False,
    "formatters": {
        "json": {
            "()": "lms.contrib.logger.format.JsonFormatter",
        },
    },
    "handlers": {
        "celery": {
            "level": "INFO",
            "class": "logging.StreamHandler",
            "formatter": "json",
        },
        "default": {
            "level": "DEBUG",
            "class": "logging.StreamHandler",
            "formatter": "json",
        },
    },
    "loggers": {
        "celery": {
            "handlers": ["celery"],
            "level": "INFO",
            "propagate": False,
        }
    },
    "root": {
        "handlers": ["default"],
        "level": LOG_LEVEL,
    }
}

# Staff
# ------------------------------------------------------------------------------
STAFF_BASE_URL = env('STAFF_BASE_URL', default='not_used')

# Staff API
# ------------------------------------------------------------------------------
STAFF_API_BASE_URL = env('STAFF_API_BASE_URL', default=None)

STAFF_API_AUTH_METHOD = env('STAFF_API_AUTH_METHOD', default=None)

STAFF_API_OAUTH_TOKEN = env('STAFF_API_OAUTH_TOKEN', default='')

STAFF_API_TVM_ID = env('STAFF_API_TVM_ID', default='staff')

STAFF_PROFILE_INACTIVE_LIMIT = env.int('STAFF_PROFILE_INACTIVE_LIMIT', default=1000)
STAFF_PROFILE_CHUNK_SIZE = env.int('STAFF_PROFILE_CHUNK_SIZE', default=100)

# Startrek
# ------------------------------------------------------------------------------
STARTREK_API_OAUTH_TOKEN = env('STARTREK_API_OAUTH_TOKEN', default='')
STARTREK_BASE_URL = env('STARTREK_BASE_URL', default='https://st.test.yandex-team.ru/')
STARTREK_API_BASE_URL = env('STARTREK_API_BASE_URL', default='https://st-api.test.yandex-team.ru/v2/')
STARTREK_EXCLUDE_FIELDS = env.list('STARTREK_EXCLUDE_FIELDS', default=[], cast=str)

STARTREK_HOOK_TOKEN = env('STARTREK_HOOK_TOKEN', default='')

# TVM
# ------------------------------------------------------------------------------
TVM_ALLOWED_CLIENTS = env.list('TVM_ALLOWED_CLIENTS', cast=int, default=[])

TVM_CHECK_CLIENTS = env.bool('TVM_CHECK_CLIENTS', default=False)

if TVM_CHECK_CLIENTS:
    TVM_API_ALLOWED_CLIENTS = env.list('TVM_API_ALLOWED_CLIENTS', default=[], cast=int)
    TVM_LABAPI_ALLOWED_CLIENTS = env.list('TVM_LABAPI_ALLOWED_CLIENTS', default=[], cast=int)


TVM_DEPLOY_SOURCE_ALIAS = env('TVM_DEPLOY_SOURCE_ALIAS', default='lms')

TVM_EXCLUDE_PATHS = [
    '/ping/',
    '/time/',
    '/admin/',
    '/static/',
    '/media/',
    '/__debug__/',
    '/tracker/',
]

TVM_DEBUG = env.bool('DJANGO_TVM_DEBUG', default=False)
TVM_ENABLED = env.bool('DJANGO_TVM_ENABLED', default=False)
# Отключает проверку сервис-тикета, если подключена TVMDebugMiddleware
DEBUG_TVM_SERVICE_ID = env.int('DJANGO_DEBUG_TVM_SERVICE_ID', default=None)
DEBUG_TVM_UID = env.int('DJANGO_DEBUG_TVM_UID', default=None)


if TVM_DEBUG:
    MIDDLEWARE += [
        'lms.contrib.tvm.middleware.TVMDebugMiddleware',
    ]
elif TVM_ENABLED:
    MIDDLEWARE += [
        'lms.contrib.tvm.middleware.TVMMiddleware',
    ]


# Blackbox / Passport
# ------------------------------------------------------------------------------
PASSPORT_URLS = {
    'desktop': {
        'create': 'https://passport-test.yandex.ru/auth/?retpath=',
        'refresh': 'https://passport-test.yandex.ru/auth/update/?retpath=',
        'passport_domain': 'passport-test.yandex.ru',
    }
}

PASSPORT_TYPE = 'desktop'

# IDM
# ------------------------------------------------------------------------------
IDM_INSTANCE = 'testing'
ROLES_HOOKS = 'lms.idm.hooks.RoleHooks'

# GUARDIAN
# ------------------------------------------------------------------------------
GUARDIAN_RAISE_403 = True
ANONYMOUS_USER_NAME = None

# BULK OPERATIONS
# ------------------------------------------------------------------------------
BULK_BATCH_SIZE_DEFAULT = 1_000


# API
# ------------------------------------------------------------------------------
API_ENABLED = env.bool("DJANGO_API_ENABLED", default=False)
API_BASE_URL = 'api/v1/'
LABAPI_BASE_URL = 'labapi/v1/'
INDEX_API_BASE_URL = 'index_api/v1/'
DATASET_BASE_URL = 'survey/datasets/'

TRACKER_API_BASE_URL = 'tracker/'

EXTERNAL_API_ENABLED = env.bool("DJANGO_EXTERNAL_API_ENABLED", default=False)
EXTERNAL_API_BASE_URL = ''  # quantum-api.yandex-team.ru


# Slugs
# ------------------------------------------------------------------------------
COURSE_RESERVED_SLUGS = env.list('COURSE_RESERVED_SLUGS', default=['new', 'archived', 'drafts', 'all'])

# Admin notice
# ------------------------------------------------------------------------------
ENVIRONMENT_NAME = env('ENVIRONMENT_NAME', default=None)
ENVIRONMENT_COLOR = env('ENVIRONMENT_COLOR', default=None)

# Simple history
# ------------------------------------------------------------------------------
SIMPLE_HISTORY_HISTORY_ID_USE_UUID = True

# Sender settings
# ------------------------------------------------------------------------------
SENDER_ENABLED = env.bool('SENDER_ENABLED', default=True)
MAILING_RECIPIENTS_CHUNK_SIZE = env.int('MAILING_RECIPIENTS_CHUNK_SIZE', default=50)

# Sender throttle times
# ------------------------------------------------------------------------------
MAIL_COURSE_AVAILABLE_AGAIN_THROTTLE_TIME = env.int('MAIL_COURSE_AVAILABLE_AGAIN_THROTTLE_TIME', default=30 * 60)
MAIL_COURSE_ENROLL_BEGIN_THROTTLE_TIME = env.int('MAIL_COURSE_ENROLL_BEGIN_THROTTLE_TIME', default=30 * 60)
MAIL_NEW_FOLLOWER_THROTTLE_TIME = env.int('MAIL_NEW_FOLLOWER_THROTTLE_TIME', default=60)

COURSE_ENROLL_THROTTLE_TIME = env.int('COURSE_ENROLL_THROTTLE_TIME', default=60 * 60 * 24)

COURSE_ENROLL_BEGIN_CHECK_FREQUENCY = env.int('COURSE_ENROLL_BEGIN_CHECK_FREQUENCY', default=30 * 60)

# Sender mailing templates
# ------------------------------------------------------------------------------
MAILING_TEMPLATES = env.json('MAILING_TEMPLATES', default={})

# UI urls
# ------------------------------------------------------------------------------
FRONTEND_ROOT = env('FRONTEND_ROOT', default='not_used')
FRONTEND_LAB_ROOT = env('FRONTEND_LAB_ROOT', default='not_used')
ADMIN_ROOT = env('ADMIN_ROOT', default='not_used')

# Cache settings

CACHE_ENABLED = env.bool('DJANGO_CACHE_ENABLED', default=False)

# Course visibility cache
COURSES_UNAVAILABLE_FOR_CACHE_KEY_TEMPLATE = 'courses_unavailable_for_{user_pk}'
COURSE_AVAILABLE_FOR_CACHE_KEY_TEMPLATE = 'course_available_for_{user_pk}_{course_pk}'

COURSE_VISIBILITY_CACHE_TTL = env.int('COURSE_VISIBILITY_CACHE_TTL', default=3_600)

# Visibility cache
STAFF_GROUP_RULE_NESTED_LIST_CACHE_TEMPLATE = 'staff_group_nested_list__{}'
STAFF_GROUP_RULE_NESTED_LIST_CACHE_TTL = 604800  # 1 week

# Cohorts
MAX_USERS_IN_COHORT = env.int('MAX_USERS_IN_COHORT', default=0)

# import export
IMPORT_EXPORT_IMPORT_PERMISSION_CODE = 'import'
IMPORT_EXPORT_EXPORT_PERMISSION_CODE = 'export'

# enrollment queues mode ('old', 'new', 'all')
ENROLLMENT_QUEUES_MODE = env('ENROLLMENT_QUEUES_MODE', default='old')

# create_tracker_enrollment_issues_bulk_task
ENROLLED_USER_TRACKER_ISSUES_AWAIT_MINUTES = env.int('ENROLLED_USER_TRACKER_ISSUES_AWAIT_MINUTES', default=30)

# Calendar API
CALENDAR_API_BASE_URL = env('CALENDAR_API_BASE_URL', default='http://host.docker.internal:8001/internal/')
CALENDAR_API_TVM_ID = env('CALENDAR_API_TVM_ID', default='calendar-api')
CALENDAR_OAUTH_TOKEN = env('CALENDAR_OAUTH_TOKEN', default='')
CALENDAR_ROBOT_UID = env('CALENDAR_ROBOT_UID', default='')
CALENDAR_API_RESPONSE_STATUS_FIELD = env('CALENDAR_API_RESPONSE_STATUS_FIELD', default='status')
CALENDAR_API_RESPONSE_SUCCESS_STATUS = env('CALENDAR_API_RESPONSE_SUCCESS_STATUS', default='ok')
CALENDAR_TIMEZONE = env('CALENDAR_TIMEZONE', default='Etc/UTC')

# Course / Module progress
COURSE_MODULE_DEFAULT_WEIGHT = env.int('COURSE_MODULE_DEFAULT_WEIGHT', default=-1)
COURSE_MODULE_WEIGHT_SUM_CACHE_TIMEOUT = env.int(
    'COURSE_MODULE_WEIGHT_SUM_CACHE_TIMEOUT', default=60 * 60 * 24
)

# Scorm
SCORM_DEFAULT_MAX_ATTEMPTS = env.int('SCORM_DEFAULT_MAX_ATTEMPTS', default=-1)
