import copy
from functools import partial

import sentry_sdk
from django_pgaas import HostManager
from django_replicated.settings import *  # noqa
from sentry_sdk.integrations.celery import CeleryIntegration

from lms.contrib.errorboosterclient.sentry import ErrorBoosterTransport
from lms.contrib.logbroker.logbroker import send_me
from lms.contrib.sentry.integrations.django import CustomDjangoIntegration as DjangoIntegration

from .base import *  # noqa
from .base import env

# GENERAL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#secret-key
SECRET_KEY = env("DJANGO_SECRET_KEY", default="lms")
# https://docs.djangoproject.com/en/dev/ref/settings/#allowed-hosts
ALLOWED_HOSTS = env.list("DJANGO_ALLOWED_HOSTS", default=["*"])

# DATABASES
# ------------------------------------------------------------------------------
POSTGRES_HOST_JSON = env.json('POSTGRES_HOST_JSON', default={})
DEPLOY_NODE_DC = env('DEPLOY_NODE_DC', default='myt')
if POSTGRES_HOST_JSON:
    host_manager = HostManager(
        list(POSTGRES_HOST_JSON.items()),
        current_dc=DEPLOY_NODE_DC,
    )
    DATABASES["default"]["HOST"] = host_manager.host_string
else:
    DATABASES["default"]["HOST"] = env('POSTGRES_HOST')

DATABASES["default"]["PORT"] = env.int('POSTGRES_PORT', default='6432')
DATABASES["default"]["ATOMIC_REQUESTS"] = True  # noqa F405
DATABASES["default"]["OPTIONS"] = {
    'sslmode': 'verify-full',
    'sslrootcert': '/app/.postgres/root.crt',
    'target_session_attrs': 'read-write',
}
DATABASES["default"]["DISABLE_SERVER_SIDE_CURSORS"] = True

DATABASES["slave"] = copy.deepcopy(DATABASES["default"])
DATABASES["slave"]["OPTIONS"]['target_session_attrs'] = 'any'

# django-replicated
# ------------------------------------------------------------------------------
REPLICATED_DATABASE_SLAVES = ['slave']
DATABASE_ROUTERS = ['django_replicated.router.ReplicationRouter']
REPLICATED_VIEWS_OVERRIDES = {
    '/admin/*': 'master',
}

# django_pgaas
# ------------------------------------------------------------------------------
PGAAS_USE_ZDM = True

# redis-sentinel
# ------------------------------------------------------------------------------
REDIS_SENTINELS = env.json("REDIS_SENTINELS", default=[])
REDIS_SERVICE_NAME = env("REDIS_SERVICE_NAME", default="lms")
REDIS_PASSWORD = env("REDIS_PASSWORD", default=None)
REDIS_CACHE_DB = env.int('REDIS_CACHE_DB', default=0)
REDIS_DB = env.int('REDIS_DB', default=1)

REDIS_MAX_CONNECTIONS = env("REDIS_MAX_CONNECTIONS", default=100)
DJANGO_REDIS_CONNECTION_FACTORY = 'django_redis.pool.SentinelConnectionFactory'

# CACHES
# ------------------------------------------------------------------------------
CACHES = {
    "default": {
        "BACKEND": "django_redis.cache.RedisCache",
        "LOCATION": "redis://{service_name}/{db}".format(
            service_name=REDIS_SERVICE_NAME,
            db=REDIS_CACHE_DB,
        ),
        "OPTIONS": {
            "CLIENT_CLASS": "django_redis.client.SentinelClient",
            "CONNECTION_POOL_KWARGS": {
                "max_connections": REDIS_MAX_CONNECTIONS,
                "retry_on_timeout": True,
            },
            "SENTINELS": REDIS_SENTINELS,
            "PASSWORD": REDIS_PASSWORD,
            # Mimicing memcache behavior.
            # https://github.com/jazzband/django-redis#memcached-exceptions-behavior
            "IGNORE_EXCEPTIONS": True,
        }
    }
}

# SECURITY
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-proxy-ssl-header
# SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-ssl-redirect
# SECURE_SSL_REDIRECT = env.bool("DJANGO_SECURE_SSL_REDIRECT", default=True)
# https://docs.djangoproject.com/en/dev/ref/settings/#session-cookie-secure
SESSION_COOKIE_SECURE = True
# https://docs.djangoproject.com/en/dev/ref/settings/#csrf-cookie-secure
CSRF_COOKIE_SECURE = True
# https://docs.djangoproject.com/en/dev/topics/security/#ssl-https
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-seconds
# TODO: set this to 60 seconds first and then to 518400 once you prove the former works
# SECURE_HSTS_SECONDS = 60
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-include-subdomains
# SECURE_HSTS_INCLUDE_SUBDOMAINS = env.bool(
#     "DJANGO_SECURE_HSTS_INCLUDE_SUBDOMAINS", default=True
# )
# https://docs.djangoproject.com/en/dev/ref/settings/#secure-hsts-preload
# SECURE_HSTS_PRELOAD = env.bool("DJANGO_SECURE_HSTS_PRELOAD", default=True)
# https://docs.djangoproject.com/en/dev/ref/middleware/#x-content-type-options-nosniff
# SECURE_CONTENT_TYPE_NOSNIFF = env.bool(
#     "DJANGO_SECURE_CONTENT_TYPE_NOSNIFF", default=True
# )

# STATIC
# ------------------------
STATICFILES_STORAGE = "whitenoise.storage.CompressedManifestStaticFilesStorage"
# MEDIA
# ------------------------------------------------------------------------------


# EMAIL
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#default-from-email
DEFAULT_FROM_EMAIL = env(
    "DJANGO_DEFAULT_FROM_EMAIL", default="lms <noreply@lms.yandex-team.ru>"
)
# https://docs.djangoproject.com/en/dev/ref/settings/#server-email
SERVER_EMAIL = env("DJANGO_SERVER_EMAIL", default=DEFAULT_FROM_EMAIL)
# https://docs.djangoproject.com/en/dev/ref/settings/#email-subject-prefix
EMAIL_SUBJECT_PREFIX = env(
    "DJANGO_EMAIL_SUBJECT_PREFIX", default="[lms]"
)

# ADMIN
# ------------------------------------------------------------------------------
ADMIN_URL = env('DJANGO_ADMIN_URL', default="admin/")

# WhiteNoise
# ------------------------------------------------------------------------------
# http://whitenoise.evans.io/en/latest/django.html#enable-whitenoise
MIDDLEWARE.insert(1, "whitenoise.middleware.WhiteNoiseMiddleware")  # noqa F405

# LOGGING
# ------------------------------------------------------------------------------
# https://docs.djangoproject.com/en/dev/ref/settings/#logging
# See https://docs.djangoproject.com/en/dev/topics/logging for
# more details on how to customize your logging configuration.

LOGGING = {
    "version": 1,
    "disable_existing_loggers": True,
    "formatters": {
        "json": {
            "()": "lms.contrib.logger.format.JsonFormatter",
        },
    },
    "handlers": {
        "console": {
            "level": "DEBUG",
            "class": "logging.StreamHandler",
            "formatter": "json",
        }
    },
    "root": {"level": LOG_LEVEL, "handlers": ["console"]},
    "loggers": {
        "": {
            "handlers": ["console"],
            "level": "WARNING",
            "propagate": False,
        },
        "django": {
            "handlers": ["console"],
            "level": "INFO",
            "propagate": False,
        },
        "django.db.backends": {
            "level": "ERROR",
            "handlers": ["console"],
            "propagate": False,
        },
        "celery": {
            "handlers": ["console"],
            "level": "INFO",
            "propagate": True,
        },
        "gunicorn.error": {
            "level": "INFO",
            "handlers": ["console"],
            "propagate": False,
        },
        "gunicorn.access": {
            "level": "INFO",
            "handlers": ["console"],
            "propagate": False,
        },
        "django_replicated": {
            "handlers": ["console"],
            "level": "INFO",
            "propagate": False,
        },
        "django_celery_beat": {
            "level": "INFO",
            "handlers": ["console"],
            "propagate": False,
        },
        # Errors logged by the SDK itself
        "sentry_sdk": {
            "level": "ERROR",
            "handlers": ["console"],
            "propagate": False,
        },
        "django.security.DisallowedHost": {
            "level": "ERROR",
            "handlers": ["console"],
            "propagate": False,
        },
    },
}

# Sentry
# ------------------------------------------------------------------------------
LOGGER = env('LOGGER', default='sentry')

# one of development|testing|prestable|production|pre-production
SENTRY_ENVIRONMENT = env('SENTRY_ENVIRONMENT', default='production')

if LOGGER == 'sentry':
    SENTRY_DSN = env('SENTRY_DSN')
    if SENTRY_DSN:
        sentry_sdk.init(
            dsn=SENTRY_DSN,
            integrations=[DjangoIntegration(), CeleryIntegration()],
            environment=SENTRY_ENVIRONMENT,
            # If you wish to associate users to errors (assuming you are using
            # django.contrib.auth) you may enable sending PII data.
            send_default_pii=True,
        )
elif LOGGER == 'error_booster':
    # Инициализируем клиент как обычно, но передаём ErrorBoosterTransport.
    LOGBROKER_TOKEN = env('LOGBROKER_TOKEN')
    LOGBROKER_SOURCE = env('LOGBROKER_SOURCE')
    LOGBROKER_TOPIC = env('LOGBROKER_TOPIC')
    ERROR_BOOSTER_PROJECT = env('ERROR_BOOSTER_PROJECT')
    LOGBROKER_DATACENTER = env('LOGBROKER_DATACENTER', default='myt')
    LOGBROKER_PORT = env.int('LOGBROKER_PORT', default=2135)

    sentry_sdk.init(
        transport=ErrorBoosterTransport(
            project=ERROR_BOOSTER_PROJECT,  # Псевдоним проекта в бустере.
            sender=partial(
                send_me,
                token=LOGBROKER_TOKEN,
                source=LOGBROKER_SOURCE,
                topic=LOGBROKER_TOPIC,
            ),
        ),
        environment=SENTRY_ENVIRONMENT,

        # Если нужно подкрутить время, выделенное
        # на работу фоновой отправляющей нити.
        shutdown_timeout=20,

        # Для интеграции с Django:
        integrations=[DjangoIntegration(), CeleryIntegration()],
        send_default_pii=True,
    )

# Deploy env
# ------------------------------------------------------------------------------
DEPLOY_STAGE_ID = env('DEPLOY_STAGE_ID', default='lms')

# Celery
# ------------------------------------------------------------------------------
if REDIS_SENTINELS:
    CELERY_BROKER_URL = ";".join(
        [
            f"sentinel://:{REDIS_PASSWORD}@{host}:{port}/{REDIS_DB}"
            for host, port in REDIS_SENTINELS
        ]
    )
    CELERY_BROKER_TRANSPORT_OPTIONS = {
        "master_name": REDIS_SERVICE_NAME,
    }

    # Beatlock
    CELERY_BEATLOCK_REDIS_URL = "redis-sentinel://"
    CELERY_BEATLOCK_REDIS_DB = env.int('CELERY_BEATLOCK_REDIS_DB', default=REDIS_DB)
    CELERY_BEATLOCK_REDIS_OPTIONS = {
        "sentinels": REDIS_SENTINELS,
        "password": REDIS_PASSWORD,
        "service_name": REDIS_SERVICE_NAME,
        "db": CELERY_BEATLOCK_REDIS_DB,
    }
    CELERY_BEATLOCK_LOCK_TIMEOUT = env("CELERY_BEATLOCK_LOCK_TIMEOUT", default=10)

else:
    CELERY_BROKER_URL = env('CELERY_BROKER_URL')


# TVM
# ------------------------------------------------------------------------------
TVM_ENABLED = env.bool('DJANGO_TVM_ENABLED', default=True)  # по умолчанию, включено

# Staff
# ------------------------------------------------------------------------------
STAFF_BASE_URL = env('STAFF_BASE_URL')

# Staff API
# ------------------------------------------------------------------------------
STAFF_API_AUTH_METHOD = env('STAFF_API_AUTH_METHOD', default='OAuth')
STAFF_API_OAUTH_TOKEN = env('STAFF_API_OAUTH_TOKEN', default='')
STAFF_API_TVM_ID = env('STAFF_API_TVM_ID', default='staff-api')


# Startrek API
# ------------------------------------------------------------------------------
STARTREK_API_OAUTH_TOKEN = env('STARTREK_API_OAUTH_TOKEN')
STARTREK_BASE_URL = env('STARTREK_BASE_URL', default='https://st.yandex-team.ru/')
STARTREK_API_BASE_URL = env('STARTREK_API_BASE_URL', default='https://st-api.yandex-team.ru/v2/')

# Blackbox / Passport
# ------------------------------------------------------------------------------
BLACKBOX_NAME = 'ProdYateam'

BLACKBOX_URL = 'http://blackbox.yandex-team.ru/blackbox'

MIDDLEWARE += [
    "lms.contrib.blackbox.middleware.AuthMiddleware",
    "lms.contrib.logger.middleware.LogUidMiddleware",
    "lms.users.middleware.UserPresenceMiddleware",
]

PASSPORT_URLS = {
    'desktop': {
        'create': 'https://passport.yandex-team.ru/auth/?retpath=',
        'refresh': 'https://passport.yandex-team.ru/auth/update/?retpath=',
        'passport_domain': 'passport.yandex-team.ru',
    }
}

PASSPORT_TYPE = 'desktop'

# YDeploy
# ------------------------------------------------------------------------------
MIDDLEWARE.insert(1, "lms.utils.middleware.FixEmptyHostMiddleware")  # noqa F405

# IDM
# ------------------------------------------------------------------------------
IDM_INSTANCE = 'production'

# S3 default bucket
# ------------------------------------------------------------------------------
AWS_ACCESS_KEY_ID = env('AWS_ACCESS_KEY_ID')
AWS_SECRET_ACCESS_KEY = env('AWS_SECRET_ACCESS_KEY')
AWS_S3_ENDPOINT_URL = env('AWS_S3_ENDPOINT_URL')
AWS_STORAGE_BUCKET_NAME = env('AWS_STORAGE_BUCKET_NAME')

# S3 upload settings
# ------------------------------------------------------------------------------
S3UPLOAD_BUCKET_ENDPOINT = env("S3UPLOAD_BUCKET_ENDPOINT")

# Sender settings
# ------------------------------------------------------------------------------
if SENDER_ENABLED:
    SENDER_AUTHORIZATION_KEY = env('SENDER_AUTHORIZATION_KEY')
    SENDER_ACCOUNT_SLUG = env('SENDER_ACCOUNT_SLUG')
    SENDER_HOST = env('SENDER_HOST')

# Sender mailing templates
# ------------------------------------------------------------------------------
MAILING_TEMPLATES = env.json('MAILING_TEMPLATES')

# UI urls
# ------------------------------------------------------------------------------
FRONTEND_ROOT = env('FRONTEND_ROOT')
FRONTEND_LAB_ROOT = env('FRONTEND_LAB_ROOT')
ADMIN_ROOT = env('ADMIN_ROOT')
