from model_utils.models import TimeStampedModel
from simple_history.models import HistoricalRecords

from django.core.exceptions import ValidationError
from django.db import models
from django.utils.translation import gettext_lazy as _

from lms.courses.models import Course, CourseModule, CourseStudent


class Assignment(CourseModule):
    @property
    def has_results(self):
        return self.results.exists()

    class Meta:
        verbose_name = _("Задание")
        verbose_name_plural = _("Задания")

    def update_completed_field(self, student: CourseStudent, value):
        AssignmentStudentResult.objects.update_or_create(
            assignment_id=self.id,
            student_id=student.id,
            defaults={'is_completed': value},
        )

    def complete(self, student: CourseStudent):
        # TODO: возможно, нужно добавить super.complete() и убрать post_save сигнал
        self.update_completed_field(student=student, value=True)

    def delete(self, *args, **kwargs):
        if self.has_results:
            raise ValidationError(_("нельзя удалить задание, по которому есть результаты"), code='invalid')

        return super(Assignment, self).delete(*args, **kwargs)


class AssignmentStudentResult(TimeStampedModel):
    assignment = models.ForeignKey(
        Assignment,
        verbose_name=_("задание"),
        related_name='results',
        on_delete=models.PROTECT,
    )
    student = models.ForeignKey(
        CourseStudent,
        verbose_name=_("студент"),
        related_name='assignment_results',
        on_delete=models.PROTECT,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='assignment_student_results',
        editable=False,
        null=True,
        blank=True,
        on_delete=models.PROTECT
    )
    is_completed = models.BooleanField(
        verbose_name=_("завершено"),
        default=False
    )

    history = HistoricalRecords()

    class Meta:
        verbose_name = _("Результат по заданию")
        verbose_name_plural = _("Результаты по заданиям")

    def clean(self):
        if self.student.course_id != self.assignment.course_id:
            raise ValidationError(
                _("курс студента должен совпадать с курсом задания"),
                code='invalid',
            )

    def save(self, *args, **kwargs):
        self.full_clean()
        if not self.course_id:
            self.course = self.assignment.course
        super().save(*args, **kwargs)
