import logging
from typing import Optional

import pytz

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction
from django.utils.dateparse import parse_datetime
from django.utils.translation import gettext_lazy as _

from lms.calendars.models import CalendarEvent, CalendarLayer
from lms.classrooms.models import StudentSlot, Timeslot
from lms.contrib.calendar.client import calendar_api
from lms.contrib.calendar.settings import CALENDAR_ROBOT_UID
from lms.courses.models import Course, CourseStudent

User = get_user_model()
log = logging.getLogger(__name__)


class CalendarException(Exception):
    pass


@transaction.atomic()
def get_or_create_calendar_layer(course_id: int):
    try:
        return CalendarLayer.objects.get(course_id=course_id)
    except CalendarLayer.DoesNotExist:
        course = Course.objects.select_for_update().select_related('author').get(pk=course_id)

        json = {
            "name": course.name,
            "participants": [{
                "email": course.author.get_staff_email(),
                "permission": "edit",
            }]
        }

        response = calendar_api.create_layer.post(json=json, uid=CALENDAR_ROBOT_UID)
        layer_id = response.get('layerId')

        if not layer_id:
            raise CalendarException(_('layerId не был получен из API календаря, курс %s') % course.id)

        return CalendarLayer.objects.create(id=layer_id, course=course)


def get_attendees(timeslot_id: int):
    slots = StudentSlot.objects.select_related('student', 'student__user').filter(
        status=StudentSlot.StatusChoices.ACCEPTED,
        timeslot_id=timeslot_id
    )

    return [slot.student.user.get_staff_email() for slot in slots]


@transaction.atomic()
def get_or_create_calendar_event(timeslot_id: int):
    try:
        return CalendarEvent.objects.get(timeslot_id=timeslot_id)
    except CalendarEvent.DoesNotExist:
        timeslot = Timeslot.objects.select_related('classroom').get(pk=timeslot_id)
        calendar_layer = get_or_create_calendar_layer(course_id=timeslot.course_id)
        classroom = timeslot.classroom

        json = {
            "layerId": calendar_layer.id,
            "name": classroom.name,
            "type": "learning",
            "availability": "busy",
            "startTs": timeslot.begin_date.replace(microsecond=0).isoformat(),
            "endTs": timeslot.end_date.replace(microsecond=0).isoformat(),
            "description": timeslot.summary,
            "attendees": get_attendees(timeslot_id=timeslot_id),
            "othersCanView": True,
        }

        response = calendar_api.create_event.post(json=json, uid=CALENDAR_ROBOT_UID)
        event_id = response.get('showEventId')

        if not event_id:
            raise CalendarException(_('showEventId не был получен из API календаря, таймслот %s') % timeslot.id)

        return CalendarEvent.objects.create(id=event_id, course_id=timeslot.course_id, timeslot=timeslot)


def get_non_blank_emails_from_list(calendar_event, property_name: str):
    item_list = calendar_event.get(property_name)
    if item_list is None:
        raise CalendarException(_(
            'Unable to find property \'%s\' in the following Calendar API response:\n%s' % (
                property_name, calendar_event
            )
        ))
    return {item.get('email') for item in item_list if item.get('email')}


def get_calendar_optional_attendees(calendar_event):
    return get_non_blank_emails_from_list(calendar_event=calendar_event, property_name='optionalAttendees')


def get_calendar_attendees(calendar_event):
    return get_non_blank_emails_from_list(calendar_event=calendar_event, property_name='attendees')


def get_calendar_meeting_rooms(calendar_event):
    return get_non_blank_emails_from_list(calendar_event=calendar_event, property_name='resources')


def is_timeslot_attendee(timeslot_id, user_id):
    return StudentSlot.objects.filter(
        status=StudentSlot.StatusChoices.ACCEPTED,
        student__user_id=user_id,
        student__status=CourseStudent.StatusChoices.ACTIVE,
        timeslot_id=timeslot_id,
    ).exists()


def update_event_via_calendar_api(
    calendar_event: CalendarEvent, begin_date, end_date, attendees=None, optional_attendees=None,
):
    if optional_attendees is None:
        optional_attendees = []
    if attendees is None:
        attendees = []

    json = {
        'attendees': sorted(attendees, key=str.casefold),
        'optionalAttendees': sorted(optional_attendees, key=str.casefold),
        'description': calendar_event.timeslot.summary,
        # В календаре эти параметры обязательные
        'startTs': begin_date,
        'endTs': end_date,
    }

    response = calendar_api.update_event.post(
        json=json,
        uid=CALENDAR_ROBOT_UID,
        id=calendar_event.id,
        tz=settings.CALENDAR_TIMEZONE,
    )
    if response.get(settings.CALENDAR_API_RESPONSE_STATUS_FIELD) != settings.CALENDAR_API_RESPONSE_SUCCESS_STATUS:
        raise CalendarException(_(
            'Unable to update event (id=%s) via Calendar API. Request payload:\n%s\nResponse:\n%s' % (
                calendar_event.id, str(json), str(response)
            )
        ))


def _update_timeslot_dates(timeslot: Timeslot, begin_date: Optional[str] = None, end_date: Optional[str] = None):
    """
    обновляет даты в таймслоте на новые, если они валидны и отличаются от текущих.
    """
    _begin_date = None if not begin_date else parse_datetime(begin_date)
    _end_date = None if not end_date else parse_datetime(end_date)
    if not _begin_date or not _end_date:
        log.warning(
            'Invalid date(s) found during timeslot update. timeslot_id=%s, begin_date=%s, end_date=%s',
            timeslot.id, begin_date, end_date,
        )
        return
    _begin_date = _begin_date.replace(tzinfo=pytz.timezone(settings.CALENDAR_TIMEZONE))
    _end_date = _end_date.replace(tzinfo=pytz.timezone(settings.CALENDAR_TIMEZONE))
    if timeslot.begin_date != _begin_date or timeslot.end_date != _end_date:
        timeslot.begin_date = _begin_date
        timeslot.end_date = _end_date
        timeslot.save()


@transaction.atomic()
def update_student_event_attendance(timeslot_id: int, student_id):
    event = CalendarEvent.objects.select_for_update().select_related(
        'timeslot'
    ).filter(timeslot_id=timeslot_id).first()

    if not event:
        log.warning(f'Unable to find CalendarEvent for timeslot_id={timeslot_id}')
        return

    user = User.objects.filter(in_courses=student_id).first()

    if not user:
        log.warning(f'Unable to find User for student_id={student_id}')
        return

    calendar_event = calendar_api.get_event.get(
        uid=CALENDAR_ROBOT_UID, eventId=event.id, tz=settings.CALENDAR_TIMEZONE,
    )
    calendar_attendees = get_calendar_attendees(calendar_event) | get_calendar_meeting_rooms(calendar_event)
    calendar_optional_attendees = get_calendar_optional_attendees(calendar_event)

    begin_date = calendar_event.get('startTs')
    end_date = calendar_event.get('endTs')
    _update_timeslot_dates(timeslot=event.timeslot, begin_date=begin_date, end_date=end_date)

    user_staff_email = user.get_staff_email()

    is_calendar_event_attendee = user_staff_email in calendar_attendees
    is_attendee = is_timeslot_attendee(timeslot_id=timeslot_id, user_id=user.id)

    if is_calendar_event_attendee == is_attendee:
        return

    attendees = calendar_attendees | {user_staff_email} if is_attendee else calendar_attendees - {user_staff_email}
    optional_attendees = calendar_optional_attendees - {user_staff_email}

    update_event_via_calendar_api(
        calendar_event=event, attendees=attendees, optional_attendees=optional_attendees,
        begin_date=begin_date, end_date=end_date,
    )


@transaction.atomic
def update_calendar_event(timeslot_id: int):
    event = CalendarEvent.objects.select_for_update().select_related(
        'timeslot'
    ).filter(timeslot_id=timeslot_id).first()

    if not event:
        log.warning(f'Unable to find CalendarEvent for timeslot_id={timeslot_id}')
        return

    calendar_event = calendar_api.get_event.get(
        uid=CALENDAR_ROBOT_UID, eventId=event.id, tz=settings.CALENDAR_TIMEZONE,
    )
    attendees = get_calendar_attendees(calendar_event) | get_calendar_meeting_rooms(calendar_event)
    optional_attendees = get_calendar_optional_attendees(calendar_event)

    begin_date = calendar_event.get('startTs')
    end_date = calendar_event.get('endTs')
    _update_timeslot_dates(timeslot=event.timeslot, begin_date=begin_date, end_date=end_date)

    update_event_via_calendar_api(
        calendar_event=event, attendees=attendees, optional_attendees=optional_attendees,
        begin_date=begin_date, end_date=end_date,
    )


def delete_calendar_event(event_id):
    response = calendar_api.delete_event.post(uid=CALENDAR_ROBOT_UID, id=event_id)
    if response.get(settings.CALENDAR_API_RESPONSE_STATUS_FIELD) != settings.CALENDAR_API_RESPONSE_SUCCESS_STATUS:
        raise CalendarException(_(
            'Unable to delete event (id=%s) via Calendar API. Response:\n%s' % (event_id, str(response))
        ))


def update_timeslot_dates_from_calendar(timeslot_id):
    event = CalendarEvent.objects.select_related(
        'timeslot'
    ).filter(timeslot_id=timeslot_id).first()

    if not event:
        return

    calendar_event = calendar_api.get_event.get(
        uid=CALENDAR_ROBOT_UID, eventId=event.id, tz=settings.CALENDAR_TIMEZONE,
    )
    begin_date = calendar_event.get('startTs')
    end_date = calendar_event.get('endTs')
    if not begin_date or not end_date:
        raise CalendarException(_(
            'Unable to find event (id=%s) dates in Calendar API response:\n%s' % (event.id, str(calendar_event))
        ))
    _update_timeslot_dates(timeslot=event.timeslot, begin_date=begin_date, end_date=end_date)
