from django_admin_listfilter_dropdown.filters import RelatedOnlyDropdownFilter
from simple_history.admin import SimpleHistoryAdmin

from django.contrib import admin
from django.utils.translation import gettext_lazy as _

from .models import Classroom, StudentSlot, Timeslot, TimeslotExchange


class TimeslotInline(admin.TabularInline):
    model = Timeslot
    fields = ('begin_date', 'end_date', 'title', 'summary', 'max_participants')
    extra = 0


@admin.register(Classroom)
class ClassroomAdmin(admin.ModelAdmin):
    list_display = ('name', 'course', 'is_active', 'created', 'modified')
    list_select_related = ('course', 'module_type')
    list_filter = (
        ('course', RelatedOnlyDropdownFilter),
    )
    raw_id_fields = ('course',)
    inlines = [
        TimeslotInline,
    ]


@admin.register(Timeslot)
class TimeslotAdmin(admin.ModelAdmin):
    list_display = ('begin_date', 'end_date', 'title', 'classroom', 'course', 'created', 'modified')
    list_select_related = ('course', 'classroom')
    list_filter = (
        ('classroom', RelatedOnlyDropdownFilter),
        ('course', RelatedOnlyDropdownFilter),
    )
    raw_id_fields = ('course', 'classroom', 'course_groups')

    fields = (
        'course', 'classroom', 'begin_date', 'end_date', 'title', 'summary',
        'num_participants', 'max_participants',
        'course_groups',
    )
    readonly_fields = ('course', 'num_participants')


@admin.register(StudentSlot)
class StudentSlotAdmin(SimpleHistoryAdmin):
    list_display = (
        'timeslot', 'status', 'display_user', 'display_group', 'display_course', 'is_attended',
        'created', 'modified',
    )
    list_select_related = (
        'timeslot',
        'timeslot__course',
        'student',
        'student__user',
        'student__group',
    )
    list_filter = (
        'status',
        ('timeslot__course', RelatedOnlyDropdownFilter),
    )
    search_fields = ('student__user__username',)
    raw_id_fields = ('timeslot', 'student')

    def display_user(self, instance: StudentSlot):
        return instance.student.user

    display_user.short_description = _("пользователь")

    def display_group(self, instance: StudentSlot):
        return instance.student.group

    display_group.short_description = _("группа")

    def display_course(self, instance: StudentSlot):
        return instance.timeslot.course

    display_course.short_description = _("курс")


@admin.register(TimeslotExchange)
class TimeslotExchangeAdmin(SimpleHistoryAdmin):
    list_display = (
        'display_user',
        'display_source_timeslot',
        'target_timeslot',
        'display_course',
        'is_active',
        'created',
        'modified',
    )
    raw_id_fields = ('student_slot', 'target_timeslot', 'course')
    search_fields = ('student_slot__student__user__username',)
    list_filter = (
        'is_active',
        ('course', RelatedOnlyDropdownFilter),
    )
    readonly_fields = ('course',)

    list_select_related = (
        'student_slot',
        'student_slot__timeslot',
        'student_slot__student',
        'student_slot__student__user',
        'target_timeslot',
        'course',
    )

    def display_user(self, instance: TimeslotExchange):
        return instance.student_slot.student.user

    display_user.short_description = _("пользователь")

    def display_source_timeslot(self, instance: TimeslotExchange):
        return instance.course

    display_source_timeslot.short_description = _("исходный слот")

    def display_course(self, instance: TimeslotExchange):
        return instance.course

    display_course.short_description = _("курс")
