from model_utils import FieldTracker
from model_utils.models import TimeStampedModel, UUIDModel
from simple_history.models import HistoricalRecords

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError
from django.db import models, transaction
from django.utils.formats import date_format
from django.utils.functional import cached_property
from django.utils.translation import gettext_lazy as _

from lms.classrooms.querysets import TimeslotExchangeQueryset
from lms.courses.mixins import CapacityQuerySetMixin
from lms.courses.models import Course, CourseGroup, CourseModule, CourseStudent

User = get_user_model()


class Classroom(CourseModule):
    calendar_enabled = models.BooleanField(_("календарь включен"), default=False)

    """
    Занятия с расписанием

    Со слотами
    """
    @property
    def can_delete(self) -> bool:
        for timeslot in self.timeslots.all():  # type: 'Timeslot'
            if not timeslot.can_delete:
                return False

        return True

    class Meta:
        verbose_name = _("Занятие c расписанием")
        verbose_name_plural = _("Занятия с расписанием")

    def delete(self, *args, **kwargs):
        if not self.can_delete:
            raise ValidationError(
                _("Нельзя удалить занятие, на которое записаны студенты."),
                code='classroom_has_timeslots',
            )
        return super().delete(*args, **kwargs)


class TimeslotQuerySet(CapacityQuerySetMixin):
    maximum_capacity_field = 'max_participants'
    current_capacity_field = 'num_participants'

    def available_for(self, user: User):
        groups_qs = CourseGroup.objects.filter(
            students__user=user,
            students__status=CourseStudent.StatusChoices.ACTIVE,
        )
        return self.filter(course_groups__in=groups_qs)


class Timeslot(TimeStampedModel):
    classroom = models.ForeignKey(
        Classroom,
        verbose_name=_("занятие"),
        related_name='timeslots',
        on_delete=models.CASCADE,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='classroom_timeslots',
        on_delete=models.PROTECT,
        blank=True,
        null=True,
        editable=False,
    )
    course_groups = models.ManyToManyField(
        CourseGroup,
        verbose_name=_("группы"),
        related_name='classroom_timeslots',
        blank=True,
    )
    begin_date = models.DateTimeField(_("дата начала"))
    end_date = models.DateTimeField(_("дата окончания"), blank=True, null=True)
    title = models.CharField(_("заголовок"), max_length=255, blank=True)
    summary = models.TextField(_("краткое описание"), blank=True)
    num_participants = models.PositiveIntegerField(_("кол-во участников"), default=0, editable=False)
    max_participants = models.PositiveIntegerField(_("макс кол-во участников"), default=0)

    objects = TimeslotQuerySet.as_manager()

    tracker = FieldTracker(fields=['begin_date', 'end_date', 'summary'])

    @cached_property
    def has_students(self) -> bool:
        statuses = StudentSlot.StatusChoices

        return self.students.filter(status=statuses.ACCEPTED).exists()

    @property
    def can_delete(self) -> bool:
        return not self.has_students

    @property
    def is_full(self) -> bool:
        return self.max_participants != 0 and self.num_participants >= self.max_participants

    class Meta:
        ordering = ('classroom', 'begin_date', '-created',)
        verbose_name = _("временной слот")
        verbose_name_plural = _("временные слоты")

    def __str__(self):
        return self.title if self.title else date_format(self.begin_date, settings.SHORT_DATETIME_FORMAT)

    def clean_end_date(self):
        if self.end_date and self.begin_date > self.end_date:
            raise ValidationError(
                {'end_date': ValidationError(_("Дата завершения слота не может быть меньше даты начала"))},
            )

    def save(self, *args, **kwargs):
        self.course = self.classroom.course
        self.clean_end_date()
        super().save(*args, **kwargs)

    def delete(self, *args, **kwargs):
        if not self.can_delete:
            raise ValidationError(
                _("Нельзя удалить слот, на который записаны студенты."),
                code='timeslot_has_students',
            )
        return super().delete(*args, **kwargs)


class StudentSlot(TimeStampedModel):
    timeslot = models.ForeignKey(
        Timeslot,
        verbose_name=_("слот"),
        related_name='students',
        on_delete=models.PROTECT,
    )
    student = models.ForeignKey(
        CourseStudent,
        verbose_name=_("студент"),
        related_name='classroom_timeslots',
        on_delete=models.PROTECT,
    )
    is_attended = models.BooleanField(_("присутствие"), default=True)

    class StatusChoices(models.TextChoices):
        ACCEPTED = 'accepted', _("зачислен")
        REJECTED = 'rejected', _("отклонен")
        CANCELED = 'canceled', _("отменен")
        EXCHANGED = 'exchanged', _("изменен")

    status = models.CharField(
        _("статус"),
        max_length=16,
        choices=StatusChoices.choices,
        default=StatusChoices.ACCEPTED,
        db_index=True,
    )

    tracker = FieldTracker(fields=['status'])

    history = HistoricalRecords()

    class Meta:
        ordering = ('timeslot', 'student', '-created')
        verbose_name = _("запись на слот")
        verbose_name_plural = _("запись на слоты")

    def __str__(self):
        return f'student: {self.student_id} timeslot: {self.timeslot_id}'

    def clean(self):
        # проверки нужны только при записи на слот
        if self.status != self.StatusChoices.ACCEPTED:
            return

        student = self.student
        timeslot = self.timeslot

        if not student.status == CourseStudent.StatusChoices.ACTIVE or not timeslot.course_id == student.course_id:
            raise ValidationError({
                'student': ValidationError(
                    _("Пользователь не является студентом курса"), code='wrong_student',
                )
            })

        timeslot_groups = set(self.timeslot.course_groups.values_list('id', flat=True))
        if timeslot_groups:
            if not student.group_id or student.group_id not in timeslot_groups:
                raise ValidationError({
                    'student': ValidationError(
                        _("Слот не доступен для текущей группы студента"), code='wrong_student_group',
                    )
                })

    @transaction.atomic
    def checkout(self):
        # проверка заполнения слота
        timeslot = Timeslot.objects.select_for_update().filter(id=self.timeslot_id).first()

        if timeslot and timeslot.is_full:
            raise ValidationError({
                'timeslot': ValidationError(
                    _("Слот не доступен для записи"), code='slot_unavailable',
                ),
            })

        # проверка наличия существующих слотов
        # TODO: добавить проверку, что можно записаться только на 1 слот в classroom в группе студента
        slots = StudentSlot.objects.filter(
            student=self.student,
            timeslot=self.timeslot,
            status=StudentSlot.StatusChoices.ACCEPTED,
        )
        if slots.exists():
            raise ValidationError(
                _("Студент уже записан на слот"), code='unique',
            )

    def _change_status(self, status: str, reason: str = None):
        if reason:
            self._change_reason = reason
        self.status = status
        self.save()

    def accept(self, reason: str = None):
        self._change_status(self.StatusChoices.ACCEPTED, reason)

    def reject(self, reason: str = None):
        self._change_status(self.StatusChoices.REJECTED, reason)

    def cancel(self, reason: str = None):
        self._change_status(self.StatusChoices.CANCELED, reason)

    def exchange(self, reason: str = None):
        self._change_status(self.StatusChoices.EXCHANGED, reason)

    def save(self, *args, **kwargs):
        self.full_clean()
        if self._state.adding:
            self.checkout()
        super().save(*args, **kwargs)


class TimeslotExchange(TimeStampedModel, UUIDModel):
    student_slot = models.ForeignKey(
        StudentSlot,
        verbose_name=_("Запись на слот"),
        related_name='exchanges',
        on_delete=models.PROTECT,
    )

    target_timeslot = models.ForeignKey(
        Timeslot,
        verbose_name=_("Целевой слот"),
        related_name='exchanges',
        on_delete=models.PROTECT,
    )

    course = models.ForeignKey(
        Course,
        null=True,
        blank=True,
        verbose_name=_("Курс"),
        related_name='timeslot_exchanges',
        on_delete=models.PROTECT,
    )

    target_student_slot = models.ForeignKey(
        StudentSlot,
        null=True,
        blank=True,
        verbose_name=_("Целевая запись на слот"),
        related_name='target_in_exchanges',
        on_delete=models.PROTECT,
    )

    exchanged_student_slot = models.ForeignKey(
        StudentSlot,
        null=True,
        blank=True,
        verbose_name=_("Новая запись на слот"),
        related_name='exchanged_in_exchanges',
        on_delete=models.PROTECT,
    )

    exchanged_target_student_slot = models.ForeignKey(
        StudentSlot,
        null=True,
        blank=True,
        verbose_name=_("Новая целевая запись на слот"),
        related_name='exchanged_target_in_exchanges',
        on_delete=models.PROTECT,
    )

    is_active = models.BooleanField(_("активен"), default=True)

    objects = TimeslotExchangeQueryset.as_manager()

    class Meta:
        verbose_name = _("обмен слотами")
        verbose_name_plural = _("обмены слотами")

    def clean(self):
        if self.student_slot.timeslot.id == self.target_timeslot.id:
            raise ValidationError(
                {
                    "target_timeslot_id": ValidationError(
                        _("Нельзя обменяться на тот же слот"),
                        code="same_timeslot"
                    )
                },
            )

        if self.student_slot.timeslot.classroom_id != self.target_timeslot.classroom_id:
            raise ValidationError(
                {
                    "target_timeslot_id": ValidationError(
                        _("Слотами можно обмениваться только в одном занятии"),
                        code="another_classroom_timeslot",
                    )
                },
            )

    def save(self, *args, **kwargs):
        self.course_id = self.target_timeslot.course_id
        self.full_clean()
        super().save(*args, **kwargs)
