from django.db import models, transaction
from django.dispatch import receiver

from lms.courses.models import CourseStudent
from lms.courses.services import refresh_course_module_weight_sum_cache

from .models import Classroom, StudentSlot, Timeslot
from .services import delete_timeslot_exchanges, refresh_timeslot_participants, update_timeslot_participants
from .tasks import reject_from_slot_on_expell_student_task


@receiver(signal=models.signals.pre_save, sender=Timeslot)
def timeslot_pre_save_handler(instance: Timeslot, **kwargs):
    if not instance._state.adding:
        refresh_timeslot_participants(timeslot=instance)


@receiver(signal=models.signals.post_save, sender=StudentSlot)
def student_slot_post_save_handler(instance: StudentSlot, created: bool, **kwargs):
    has_changed = instance.tracker.has_changed('status')
    previous = instance.tracker.previous('status')
    current = instance.status
    statuses = StudentSlot.StatusChoices

    if (created or has_changed) and instance.status == statuses.ACCEPTED:
        update_timeslot_participants(instance.timeslot, value=1)

    if has_changed and previous == statuses.ACCEPTED and current != statuses.ACCEPTED:
        update_timeslot_participants(instance.timeslot, value=-1)

    if has_changed and previous == statuses.ACCEPTED and current in [statuses.CANCELED, statuses.REJECTED]:
        delete_timeslot_exchanges(instance)


@receiver(signal=models.signals.post_delete, sender=StudentSlot)
def student_slot_post_delete_handler(instance: StudentSlot, **kwargs):
    if instance.status == StudentSlot.StatusChoices.ACCEPTED:
        update_timeslot_participants(instance.timeslot, value=-1)


@receiver(signal=models.signals.post_save, sender=CourseStudent)
def course_student_post_save_handler(instance: CourseStudent, created, **kwargs):
    if (
        not created and
        instance.tracker.has_changed('status') and
        instance.status == CourseStudent.StatusChoices.EXPELLED
    ):
        transaction.on_commit(lambda: reject_from_slot_on_expell_student_task.delay(student_id=instance.id))


@receiver(signal=models.signals.post_save, sender=Classroom)
def classroom_post_save_handler(instance: Classroom, created: bool, **kwargs):
    if created or instance.field_tracker.has_changed('weight') or instance.field_tracker.has_changed('is_active'):
        transaction.on_commit(lambda: refresh_course_module_weight_sum_cache(course_id=instance.course_id))


@receiver(signal=models.signals.post_delete, sender=Classroom)
def classroom_post_delete_handler(instance: Classroom, **kwargs):
    transaction.on_commit(lambda: refresh_course_module_weight_sum_cache(course_id=instance.course_id))
