import warnings

from drf_spectacular.utils import OpenApiParameter, extend_schema

from django.core.exceptions import ValidationError as DjangoValidationError
from django.db import transaction
from django.db.models import Prefetch
from django.utils.translation import gettext
from django.utils.translation import gettext_lazy as _

from rest_framework import status
from rest_framework.exceptions import ValidationError
from rest_framework.fields import BooleanField
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.serializers import as_serializer_error

from lms.core.views.pagination import LimitOffsetAllPagination
from lms.core.views.viewsets import APIModelViewSet
from lms.courses.models import CourseStudent
from lms.courses.views.api import CourseModuleBaseViewSet

from ..mixins import StudentSlotMixin
from ..models import Classroom, StudentSlot, Timeslot, TimeslotExchange
from ..serializers import (
    ClassroomDetailSerializer, ClassroomListSerializer, StudentSlotDetailSerializer, StudentSlotListSerializer,
    TimeslotChangeSerializer, TimeslotDetailSerializer, TimeslotExchangeDetailSerializer, TimeslotExchangeSerializer,
    TimeslotExchangeUpdateSerializer, TimeslotListSerializer,
)
from ..services import accept_exchange, change_timeslot

AVAILABLE_SLOTS_PARAMETER = OpenApiParameter(
    name='available_slots',
    description=gettext("Показать только доступные слоты для пользователя"),
    required=False,
    type=bool,
)


class ClassroomViewSet(CourseModuleBaseViewSet):
    """
    Занятия с расписанием
    """
    serializer_class = ClassroomListSerializer
    serializer_classes = {
        'list': ClassroomListSerializer,
        'retrieve': ClassroomDetailSerializer
    }
    queryset = Classroom.objects.active()
    pagination_class = LimitOffsetAllPagination

    @property
    def is_timeslots_available(self):
        value = self.request.query_params.get('available_slots')
        if value:
            return BooleanField().to_representation(value)
        return False

    def get_queryset(self):
        queryset = super().get_queryset()
        user = getattr(self.request, 'user')
        prefetched = ['timeslots']

        if self.action == 'list':
            filter_kwargs = {'course_id': self.get_course().id}
            queryset = queryset.filter(**filter_kwargs)

        # фильтр по доступным слотам применяется для списка и одного занятия
        if self.action in ['list', 'retrieve']:
            if user and self.is_timeslots_available:
                prefetched = [
                    Prefetch('timeslots', queryset=Timeslot.objects.available_for(user)),
                ]

        if prefetched:
            queryset = queryset.prefetch_related(*prefetched)

        return queryset

    @extend_schema(
        parameters=[
            AVAILABLE_SLOTS_PARAMETER
        ],
        summary=gettext("Список занятий с расписанием")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        parameters=[
            AVAILABLE_SLOTS_PARAMETER,
        ],
        summary=gettext("Информация о занятии")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class ClassroomTimeslotViewSet(CourseModuleBaseViewSet):
    """
    Слоты для занятий с расписанием
    """
    queryset = Timeslot.objects.all()
    serializer_class = TimeslotListSerializer
    serializer_classes = {
        'list': TimeslotListSerializer,
        'retrieve': TimeslotDetailSerializer,
    }
    pagination_class = LimitOffsetAllPagination

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            filter_kwargs = {'course_id': self.get_course().id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset

    @extend_schema(
        summary=gettext("Список слотов для занятия с расписанием")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Информация о слоте")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class ClassroomTimeslotCheckinViewSet(APIModelViewSet):
    """
    Запись на слот
    """
    queryset = Timeslot.objects.all()
    serializer_class = StudentSlotDetailSerializer
    permission_classes = [
        IsAuthenticated,
    ]

    _cached_timeslot = None

    def get_timeslot(self) -> 'Timeslot':
        if not self._cached_timeslot:
            queryset = super().get_queryset()
            timeslot_id = self.kwargs['pk']
            self._cached_timeslot = get_object_or_404(queryset, pk=timeslot_id)

        return self._cached_timeslot

    def get_current_student(self):
        return get_object_or_404(
            CourseStudent.objects.active(),
            user=self.request.user,
            course_id=self.get_timeslot().course_id
        )

    @extend_schema(
        request=None,
        summary=gettext("Создать запись на слот")
    )
    def create(self, request, *args, **kwargs):
        try:
            instance, _ = StudentSlot.objects.get_or_create(
                timeslot=self.get_timeslot(),
                student=self.get_current_student(),
                status=StudentSlot.StatusChoices.ACCEPTED,
            )
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))

        response_serializer = self.get_retrieve_serializer(instance)
        return Response(response_serializer.data, status=status.HTTP_201_CREATED)


class ClassroomTimeslotCancelViewSet(StudentSlotMixin, APIModelViewSet):
    """
    Отмена записи на слот
    """
    queryset = StudentSlot.objects.all()
    serializer_class = StudentSlotDetailSerializer
    permission_classes = [
        IsAuthenticated,
    ]

    @extend_schema(request=None, summary=gettext("Отменить запись на слот"))
    def create(self, request, *args, **kwargs):
        student_slot = self.get_object()

        student_slot.cancel(_("Заявка отменена студентом"))

        response_serializer = self.get_retrieve_serializer(student_slot)
        return Response(response_serializer.data, status=status.HTTP_200_OK)


class ClassroomTimeslotChangeViewSet(StudentSlotMixin, APIModelViewSet):
    """
    Перезаписаться в слот, в котором есть свободные места
    """
    queryset = StudentSlot.objects.select_related('timeslot', 'student')
    serializer_classes = {
        'create': TimeslotChangeSerializer,
        'retrieve': StudentSlotDetailSerializer,
    }
    permission_classes = [
        IsAuthenticated,
    ]

    @extend_schema(
        summary=gettext("Перезаписаться в слот, в котором есть свободные места"),
        responses={201: StudentSlotDetailSerializer}
    )
    @transaction.atomic
    def create(self, request, *args, **kwargs):
        try:
            new_student_slot = change_timeslot(
                student_slot=self.get_object(),
                target_timeslot_id=self.get_target_timeslot_id(request),
            )
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))

        response_serializer = self.get_retrieve_serializer(new_student_slot)
        return Response(response_serializer.data, status=status.HTTP_201_CREATED)

    def get_target_timeslot_id(self, request) -> int:
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid()

        return serializer.data['target_timeslot_id']


class TimeslotExchangeCreateViewSet(StudentSlotMixin, APIModelViewSet):
    queryset = StudentSlot.objects.all()
    serializer_classes = {
        'update': TimeslotExchangeUpdateSerializer,
        'retrieve': TimeslotExchangeSerializer,
    }
    permission_classes = [
        IsAuthenticated,
    ]

    @extend_schema(
        summary=gettext("Создать запрос на обмен слотами"),
        responses={201: TimeslotExchangeSerializer},
    )
    def update(self, request, *args, **kwargs):
        try:
            slot_exchange, _ = TimeslotExchange.objects.get_or_create(
                student_slot=self.get_object(),
                target_timeslot_id=self._get_target_timeslot_id(request),
                is_active=True,
            )
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))

        response_serializer = self.get_retrieve_serializer(slot_exchange)
        return Response(response_serializer.data, status=status.HTTP_201_CREATED)

    def _get_target_timeslot_id(self, request):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        return serializer.data['target_timeslot_id']


class TimeslotExchangeViewSet(APIModelViewSet):
    queryset = TimeslotExchange.objects.active().select_related(
        'student_slot', 'student_slot__timeslot', 'target_timeslot'
    )
    serializer_class = TimeslotExchangeDetailSerializer

    def get_queryset(self):
        course_ids = CourseStudent.objects.filter(
            user=self.request.user,
            course__is_active=True,
            course__is_archive=False,
        ).values_list('course_id', flat=True)

        return super().get_queryset().filter(course_id__in=course_ids)

    @extend_schema(summary=gettext("Информация об обмене слотами"))
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class TimeslotExchangeAcceptViewSet(APIModelViewSet):
    queryset = TimeslotExchange.objects.active().select_related(
        'student_slot',
        'student_slot__timeslot',
        'target_timeslot',
    )
    target_slot_queryset = StudentSlot.objects.select_related(
        'timeslot',
    )
    serializer_class = StudentSlotDetailSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        queryset = super().get_queryset()

        if self.action == 'create':
            queryset = queryset.select_for_update(of=('self',))

        return queryset

    def get_target_student_slot(self, target_timeslot_id: int):
        try:
            return self.target_slot_queryset.get(
                timeslot_id=target_timeslot_id,
                student__user=self.request.user,
                status=StudentSlot.StatusChoices.ACCEPTED
            )
        except StudentSlot.DoesNotExist:
            raise ValidationError(_('Нет записи на занятие'), code='no_student_slot')

    @extend_schema(request=None, summary=gettext("Принять запрос на обмен слотами"))
    @transaction.atomic
    def create(self, request, *args, **kwargs):
        exchange = self.get_object()
        target_student_slot = self.get_target_student_slot(exchange.target_timeslot_id)

        try:
            accept_exchange(exchange, target_student_slot)
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))

        response_serializer = self.get_retrieve_serializer(exchange.exchanged_target_student_slot)
        return Response(response_serializer.data, status=status.HTTP_201_CREATED)


class StudentClassroomTimeslotViewSet(APIModelViewSet):
    """
    Управление слотами студента
    """
    queryset = StudentSlot.objects.filter(status=StudentSlot.StatusChoices.ACCEPTED)
    serializer_class = StudentSlotListSerializer
    serializer_classes = {
        'retrieve': StudentSlotDetailSerializer,
        'list': StudentSlotListSerializer,
    }
    pagination_class = LimitOffsetAllPagination
    permission_classes = [
        IsAuthenticated,
    ]

    # def get_classroom(self) -> 'Classroom':
    #     queryset = Classroom.objects.select_related('course').active()
    #     if self.action == 'list':
    #         classroom_id = self.kwargs['pk']
    #         filter_kwargs = {'pk': classroom_id}
    #     if self.action == 'destroy':
    #         timeslot_id = self.kwargs['pk']
    #         filter_kwargs = {'timeslots__id': timeslot_id}
    #     return get_object_or_404(queryset, **filter_kwargs)
    #
    # def get_course(self) -> 'Course':
    #     return self.get_classroom().course

    def get_queryset(self):
        queryset = super().get_queryset()
        user = getattr(self.request, 'user', None)

        if not user:
            return queryset.none()

        filter_kwargs = {}
        if self.action == 'list':
            classroom_id = self.kwargs['pk']
            filter_kwargs = {
                'timeslot__classroom_id': classroom_id,
                'student__user': user,
                'student__status': CourseStudent.StatusChoices.ACTIVE,
            }

        if self.action == 'destroy':
            warnings.warn('ручка сломана, но не используется (get по timeslot_id вместо studentslot_id)', FutureWarning)
            timeslot_id = self.kwargs['pk']
            filter_kwargs = {
                'timeslot_id': timeslot_id,
                'student__user': user,
                'student__status': CourseStudent.StatusChoices.ACTIVE,
            }

        if not filter_kwargs:
            return queryset.none()

        return queryset.filter(**filter_kwargs)

    @extend_schema(
        summary=gettext("Список слотов, на которые записан студент")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        responses={200: StudentSlotDetailSerializer},
        summary=gettext("Информация по слоту")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление записи на слот"),
    )
    def destroy(self, request, *args, **kwargs):
        warnings.warn('ручка сломана, но не используется (get по timeslot_id вместо studentslot_id)', FutureWarning)
        return super().destroy(request, *args, **kwargs)

    def perform_destroy(self, instance):
        try:
            return instance.delete()
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))
