from drf_spectacular.utils import extend_schema

from django.core.exceptions import ValidationError as DjangoValidationError
from django.shortcuts import get_object_or_404
from django.utils.translation import gettext

from rest_framework.exceptions import ValidationError
from rest_framework.serializers import as_serializer_error

from lms.core.views.pagination import LimitOffsetAllPagination
from lms.core.views.viewsets import LabModelViewSet
from lms.courses.permissions import CourseObjectPermission
from lms.courses.views import GetCourseMixin

from ..models import Classroom, StudentSlot, Timeslot
from ..serializers import (
    ClassroomCreateLabSerializer, ClassroomDetailLabSerializer, ClassroomDetailNoTimeslotsLabSerializer,
    ClassroomListLabSerializer, ClassroomUpdateLabSerializer, StudentSlotListLabSerializer, TimeslotCreateLabSerializer,
    TimeslotDetailLabSerializer, TimeslotUpdateLabSerializer,
)


# TODO: выпилить методы create, update, partial_update, retrieve (перенесено в tracker/views/labapi.py)
class ClassroomLabViewSet(GetCourseMixin, LabModelViewSet):
    """
    Занятия с расписанием
    """
    serializer_class = ClassroomListLabSerializer
    serializer_classes = {
        'list': ClassroomListLabSerializer,
        'create': ClassroomCreateLabSerializer,
        'update': ClassroomUpdateLabSerializer,
        'partial_update': ClassroomUpdateLabSerializer,
        'retrieve': ClassroomDetailLabSerializer,
    }
    queryset = Classroom.objects.select_related('course').prefetch_related('timeslots', 'timeslots__course_groups')
    pagination_class = LimitOffsetAllPagination
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            filter_kwargs = {'course_id': course_id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset

    def get_retrieve_serializer(self, *args, **kwargs):
        kwargs['context'] = self.get_serializer_context()
        return ClassroomDetailNoTimeslotsLabSerializer(*args, **kwargs)

    @extend_schema(
        responses={200: ClassroomDetailLabSerializer},
        summary=gettext("Информация по занятий с расписанием")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        request=ClassroomCreateLabSerializer,
        responses={201: ClassroomDetailLabSerializer},
        summary=gettext("Создание занятия с расписанием"),
    )
    def create(self, request, *args, **kwargs):
        return super().create(request, *args, **kwargs)

    @extend_schema(
        request=ClassroomUpdateLabSerializer,
        responses={200: ClassroomDetailLabSerializer},
        summary=gettext("Обновление занятия с расписанием"),
    )
    def update(self, request, *args, **kwargs):
        return super().update(request, *args, **kwargs)

    @extend_schema(
        request=ClassroomUpdateLabSerializer,
        responses={200: ClassroomDetailLabSerializer},
        summary=gettext("Частичное обновление занятия с расписанием"),
    )
    def partial_update(self, request, *args, **kwargs):
        return super().partial_update(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление занятия с расписанием"),
    )
    def destroy(self, request, *args, **kwargs):
        return super().destroy(request, *args, **kwargs)


class ClassroomTimeslotLabViewSet(LabModelViewSet):
    """
    Слоты для занятий с расписанием
    """
    serializer_class = TimeslotDetailLabSerializer
    serializer_classes = {
        'create': TimeslotCreateLabSerializer,
        'update': TimeslotUpdateLabSerializer,
        'partial_update': TimeslotUpdateLabSerializer,
        'retrieve': TimeslotDetailLabSerializer
    }
    queryset = Timeslot.objects.select_related('classroom', 'course')
    pagination_class = LimitOffsetAllPagination
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]

    _cached_classroom = None

    def get_classroom(self, pk) -> 'Classroom':
        if not self._cached_classroom:
            queryset = Classroom.objects.select_related('course')
            self._cached_classroom = get_object_or_404(queryset, pk=pk)
        return self._cached_classroom

    def get_course(self, obj=None):
        if obj is not None:
            return obj.course

        if self.action == 'create':
            return self.get_classroom(self.request.data.get('classroom_id')).course

        if self.action == 'list':
            return self.get_classroom(self.kwargs.get('pk')).course

    def get_queryset(self):
        queryset = super().get_queryset()

        if self.action == 'list':
            classroom_id = self.kwargs['pk']
            filter_kwargs = {'classroom_id': classroom_id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset

    def get_serializer_context(self):
        context = super().get_serializer_context()
        if self.action == 'create':
            context['course'] = self.get_course()
        return context

    @extend_schema(
        responses={200: TimeslotDetailLabSerializer},
        summary=gettext("Информация по слоту")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        request=TimeslotCreateLabSerializer,
        responses={201: TimeslotDetailLabSerializer},
        summary=gettext("Создание слота для занятия"),
    )
    def create(self, request, *args, **kwargs):
        return super().create(request, *args, **kwargs)

    @extend_schema(
        request=TimeslotUpdateLabSerializer,
        responses={200: TimeslotDetailLabSerializer},
        summary=gettext("Обновление слота"),
    )
    def update(self, request, *args, **kwargs):
        return super().update(request, *args, **kwargs)

    @extend_schema(
        request=TimeslotUpdateLabSerializer,
        responses={200: TimeslotDetailLabSerializer},
        summary=gettext("Частичное обновление слота"),
    )
    def partial_update(self, request, *args, **kwargs):
        return super().partial_update(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление слота"),
    )
    def destroy(self, request, *args, **kwargs):
        return super().destroy(request, *args, **kwargs)

    def perform_update(self, instance):
        try:
            return super().perform_update(instance)
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))

    def perform_destroy(self, instance):
        try:
            return instance.delete()
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))


class ClassroomTimeslotStudentListLabViewSet(LabModelViewSet):
    """
    Список студентов, записанных на слот
    """
    queryset = StudentSlot.objects.select_related(
        'timeslot', 'timeslot__course', 'student', 'student__user', 'student__group')
    serializer_class = StudentSlotListLabSerializer
    pagination_class = LimitOffsetAllPagination
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]
    lookup_url_kwarg = 'pk'
    lookup_field = 'timeslot_id'

    _cached_timeslot = None

    def get_timeslot(self, pk) -> 'Timeslot':
        if not self._cached_timeslot:
            queryset = Timeslot.objects.select_related('course')
            self._cached_timeslot = get_object_or_404(queryset, pk=pk)
        return self._cached_timeslot

    def get_course(self, obj=None):
        if obj is not None:
            return obj.timeslot.course

        if self.action == 'list':
            return self.get_timeslot(self.kwargs.get('pk')).course

    @extend_schema(
        summary=gettext("Список студентов, записанных на слот")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)
