import logging

import blackbox

from django.contrib.auth import get_user_model
from django.contrib.auth.middleware import AuthenticationMiddleware
from django.utils.functional import SimpleLazyObject

from .blackbox import BlackboxTVM2
from .settings import (
    BLACKBOX_ATTRIBUTES, BLACKBOX_CREATE_USER_ON_DEMAND, BLACKBOX_EXTRA_KWARGS, BLACKBOX_GET_USER_TICKET, BLACKBOX_NAME,
    BLACKBOX_SESSIONID2_COOKIE_NAME, BLACKBOX_SESSIONID2_REQUIRED, BLACKBOX_SESSIONID_COOKIE_NAME, BLACKBOX_URL,
    BLACKBOX_USER_YAUID_FIELD_NAME,
)
from .utils import get_attributes, get_current_host, get_default_email, get_real_ip, get_uid, is_secure, is_valid

try:
    from ticket_parser2_py3.api.v1 import BlackboxClientId, BlackboxEnv
except ImportError:
    from ticket_parser2.api.v1 import BlackboxClientId, BlackboxEnv  # noqa


log = logging.getLogger(__name__)

User = get_user_model()


class AuthException(Exception):
    pass


class AuthMiddleware(AuthenticationMiddleware):
    def process_request(self, request):
        if not hasattr(request, 'user'):
            request.user = SimpleLazyObject(lambda: self.get_user(request))

    def get_user(self, request):
        from django.contrib.auth.models import AnonymousUser

        if not hasattr(request, '_cached_user'):
            user = None
            session = self.get_session(request)

            if is_valid(session):
                try:
                    user = User.objects.get(**{User.USERNAME_FIELD: get_attributes(session).get('login')})
                except User.DoesNotExist:
                    if BLACKBOX_CREATE_USER_ON_DEMAND:
                        user = self.create_user(session)

            request._cached_user = user or AnonymousUser()

        return request._cached_user

    def create_user(self, session):
        defaults = {
            'email': get_default_email(session) or '',
            BLACKBOX_USER_YAUID_FIELD_NAME: get_uid(session),
        }

        create_kwargs = {
            User.USERNAME_FIELD: get_attributes(session).get('login'),
            'defaults': defaults,
        }

        user, created = User.objects.get_or_create(**create_kwargs)
        if created:
            user.set_unusable_password()
            user.save()

        return user

    def get_session(self, request):
        log.debug('blackbox get_session')
        if not hasattr(request, '_passport_session'):
            session_id = self.get_session_id(request)
            if session_id is None:
                return None

            bb = BlackboxTVM2(
                url=BLACKBOX_URL,
                blackbox_client=BlackboxClientId[BLACKBOX_NAME],
            )

            try:
                userip = get_real_ip(request)
                server_host = get_current_host(request)

                session = bb.sessionid(
                    sessionid=session_id,
                    userip=userip,
                    host=server_host,
                    attributes=",".join(BLACKBOX_ATTRIBUTES.keys()),
                    sslsessionid=self.get_session_id2(request),
                    get_user_ticket='yes' if BLACKBOX_GET_USER_TICKET else None,
                    **BLACKBOX_EXTRA_KWARGS,
                )

                error = session.get('error', '')
                if error != 'OK':
                    log.warning('Blackbox responded with error: %s. Request was from ip %s, host %s.',
                                error, userip, server_host)

            except (blackbox.BlackboxResponseError, blackbox.BlackboxError):
                log.error('Blackbox error', exc_info=True)
                return None

            if BLACKBOX_SESSIONID2_REQUIRED and not is_secure(session):
                raise AuthException('Session is not secure')

            request._passport_session = session

        return request._passport_session

    def get_session_id(self, request):
        return request.COOKIES.get(BLACKBOX_SESSIONID_COOKIE_NAME)

    def get_session_id2(self, request):
        if not BLACKBOX_SESSIONID2_REQUIRED:
            return

        if not request.is_secure():
            log.error('Authorization error: HTTPS connection is required when using "sessionid2"')
            raise AuthException('HTTPS connection is required')

        session_id2 = request.COOKIES.get(BLACKBOX_SESSIONID2_COOKIE_NAME)

        if not session_id2 and BLACKBOX_SESSIONID2_REQUIRED:
            raise AuthException('No sessionid2 cookie, but is required')

        return session_id2
