import logging
from urllib.parse import urljoin

import requests
from blackbox import BlackboxError, BlackboxResponseError
from requests.auth import AuthBase
from tvm2.protocol import BlackboxClientId

from django.core.exceptions import ImproperlyConfigured
from django.utils.functional import SimpleLazyObject

from lms.contrib.blackbox.blackbox import BlackboxTVM2
from lms.contrib.blackbox.settings import BLACKBOX_NAME, BLACKBOX_URL

from .settings import CALENDAR_API_BASE_URL, CALENDAR_API_SKIP_TEST, CALENDAR_API_TVM_ID, CALENDAR_OAUTH_TOKEN

log = logging.getLogger(__name__)


class CalendarTVMAuth(AuthBase):
    service_ticket_header = 'X-Ya-Service-Ticket'
    user_ticket_header = 'X-Ya-User-Ticket'

    def __init__(self, tvm2_id, tvm2_client, oauth_token=None, retries=3):
        self.oauth_token = oauth_token
        self.tvm2_id = tvm2_id
        self.tvm2_client = tvm2_client
        self.retry_attempts = retries
        self.bb = BlackboxTVM2(url=BLACKBOX_URL, blackbox_client=BlackboxClientId[BLACKBOX_NAME])

    def __call__(self, request):
        self._set_service_ticket_header(request)
        self._set_user_ticket_header(request)

        return request

    def _set_service_ticket_header(self, request):
        for _ in range(self.retry_attempts):
            service_tickets = self.tvm2_client.get_service_tickets(self.tvm2_id)
            calendar_ticket = service_tickets.get(self.tvm2_id)
            if calendar_ticket:
                request.headers[self.service_ticket_header] = calendar_ticket
                break
        else:
            log.error('Cannot get service ticket for Calendar API')

    def _set_user_ticket_header(self, request):
        if not self.oauth_token:
            return

        for _ in range(self.retry_attempts):
            try:
                passport_user = self.bb.oauth(
                    oauth_token=self.oauth_token,
                    userip='127.0.0.1',
                    get_user_ticket='yes',
                    by_token=True,
                )

                error = passport_user.get('error', '')
                if error != 'OK':
                    log.warning('Blackbox responded with error: %s.', error)

            except (BlackboxResponseError, BlackboxError):
                log.error('Blackbox error', exc_info=True)
                continue

            user_ticket = passport_user.get('user_ticket')
            if user_ticket:
                request.headers[self.user_ticket_header] = user_ticket
                break
        else:
            log.error('Cannot get user ticket for Calendar API')


class Endpoint:
    def __init__(self, client: 'CalendarClient', name: str, **kwargs):
        self.client = client
        self.name = name

    def request(self, **kwargs):
        return self.client.api_request(self.name, **kwargs)

    def get(self, **kwargs):
        return self.request(params=kwargs)

    def post(self, json=None, **kwargs):
        return self.request(method='post', json=json, params=kwargs)


class CalendarClient:
    def __init__(self, base_url=None, timeout=10, retries=3, session=None, **kwargs):
        self.base_url = base_url or CALENDAR_API_BASE_URL
        self.timeout = timeout
        self.retries = retries
        self.session = session or requests.Session()

    def api_request(self, endpoint, method='get', **kwargs):
        url = urljoin(self.base_url, endpoint)

        if CALENDAR_API_SKIP_TEST:
            return {}

        response = self.session.request(method, url, timeout=self.timeout, **kwargs)
        response.raise_for_status()
        return self.parse_response(response)

    def parse_response(self, response):
        return response.json()

    def __getattr__(self, item):
        item = item.replace('_', '-')

        return type(item, (Endpoint,), {})(client=self, name=item)


def get_client() -> CalendarClient:
    session = requests.session()

    if not CALENDAR_API_TVM_ID:
        raise ImproperlyConfigured('Please set the CALENDAR_API_TVM_ID setting')

    from lms.contrib.tvm.client import tvm2_client
    session.auth = CalendarTVMAuth(
        tvm2_id=CALENDAR_API_TVM_ID,
        tvm2_client=tvm2_client,
        oauth_token=CALENDAR_OAUTH_TOKEN
    )

    return CalendarClient(session=session)


calendar_api = SimpleLazyObject(lambda: get_client())
