import os.path
from multiprocessing.pool import ThreadPool
from typing import Iterable

import boto3
from botocore.exceptions import ClientError

from django.conf import settings
from django.utils.functional import SimpleLazyObject

from .settings import S3_UNPACK_UPLOAD_THREADS


class S3BotoClient:
    def __init__(
        self,
        aws_access_key_id: str,
        aws_secret_access_key: str,
        endpoint_url: str,
        bucket: str,
    ):
        self.aws_access_key_id = aws_access_key_id
        self.aws_secret_access_key = aws_secret_access_key
        self.endpoint_url = endpoint_url
        self.bucket = bucket

        self.session = boto3.session.Session(
            aws_access_key_id=self.aws_access_key_id,
            aws_secret_access_key=self.aws_secret_access_key,
        )

        self.s3 = self.session.client(
            service_name='s3',
            endpoint_url=self.endpoint_url,
        )

    def file_exists(self, s3_file_path: str) -> bool:
        """
        Проверка существования файла на s3
        :param s3_file_path: путь на s3 к проверемому файлу
        """
        try:
            self.s3.head_object(Bucket=self.bucket, Key=s3_file_path)
        except ClientError as e:
            if int(e.response['Error']['Code']) == 404:
                return False
            raise

        return True

    def download_file(self, s3_file_path: str, local_file_path: str) -> None:
        """
        Загружзка файла с s3
        :param s3_file_path: путь на s3 к загружаемому файлу
        :param local_file_path: локальный путь к загружаемому файлу
        """
        if not self.file_exists(s3_file_path):
            raise Exception("File %s not found in s3" % s3_file_path)

        self.s3.download_file(self.bucket, s3_file_path, local_file_path)

    def upload_file(self, local_file_path: str, s3_file_path: str) -> None:
        """
        Загрузка файла на s3
        :param local_file_path: локальный путь к загружаемому файлу
        :param s3_file_path: путь на s3 к загружаемому каталогу
        """
        with open(local_file_path, 'rb') as stream:
            self.s3.put_object(Bucket=self.bucket, Key=s3_file_path, Body=stream)

    def upload_folder(self, local_root_path: str, s3_root_path: str) -> None:
        """
        Рекурсивная загрузка каталога на s3
        :param local_root_dir: локальный путь к загружаемому каталогу
        :param s3_root_path: путь на s3 к загружаемому каталогу
        """
        local_root_path = local_root_path.rstrip('/')
        local_root_path_len = len(local_root_path)

        # рекурсивный обход файлов в каталоге
        def walk_with_files(folder) -> Iterable[str]:
            for dirpath, dirnames, filenames in os.walk(folder):
                for filename in filenames:
                    yield os.path.join(dirpath, filename)  # полный путь к локальному файлу

        # загрузка файла на s3
        def upload_one_file(local_full_file_path) -> None:
            file_path = local_full_file_path[local_root_path_len + 1:]  # относительный путь к локальному файлу
            s3_file_path = os.path.join(s3_root_path, file_path)
            self.upload_file(
                local_file_path=local_full_file_path,
                s3_file_path=s3_file_path,
            )

        pool = ThreadPool(processes=S3_UNPACK_UPLOAD_THREADS)
        pool.map(upload_one_file, walk_with_files(local_root_path))

    def generate_presigned_url(self, s3_file_path: str):
        return self.s3.generate_presigned_url(
            "put_object",
            Params={
                "Bucket": self.bucket,
                "Key": s3_file_path,
            },
            ExpiresIn=3600,
            HttpMethod='PUT',
        )


def get_client(bucket_name: str = None) -> S3BotoClient:
    if bucket_name is None:
        bucket_name = settings.AWS_STORAGE_BUCKET_NAME

    return S3BotoClient(
        aws_access_key_id=settings.AWS_ACCESS_KEY_ID,
        aws_secret_access_key=settings.AWS_SECRET_ACCESS_KEY,
        endpoint_url=settings.AWS_S3_ENDPOINT_URL,
        bucket=bucket_name,
    )


s3_boto_client = SimpleLazyObject(lambda: get_client())
