import logging

from django.core.exceptions import ValidationError as DjangoValidationError
from django.db import models
from django.utils.translation import gettext_lazy as _

from rest_framework import serializers, status
from rest_framework.exceptions import ValidationError
from rest_framework.fields import BooleanField
from rest_framework.response import Response

from .permissions import ApiAllowedTVMServicePermission, LabApiAllowedTVMServicePermission, LabPermissions

log = logging.getLogger(__name__)


class ActionSerializerMixin:
    """
    Миксин для ViewSet, который переопределяет serializer
    в зависимости от action

    Пример,

    from rest_framework.viewsets import ModelViewSet

    class ExampleViewSet(ModelViewSet):
        serializer_class = ListSerializer
        serializer_classes = {
            'create': CreateSerializer,
            'retrieve': DetailSerializer,
            'update': UpdateSerializer,
            'list': ListSerializer,
        }

    Поле serializer_class используется также, как дефолтный serializer

    """
    serializer_classes = None

    def get_serializer_class(self):
        serializer_classes = self.serializer_classes or {}
        return serializer_classes.get(
            self.action,
            self.serializer_class
        )

    def get_retrieve_serializer(self, *args, **kwargs):
        serializer_classes = self.serializer_classes or {}
        serializer_class = serializer_classes.get('retrieve', self.serializer_class)
        kwargs['context'] = self.get_serializer_context()
        return serializer_class(*args, **kwargs)


class CreateWithRetrieveModelMixin:
    """
    Создание модели с ответом как в get-ручке
    """

    def create_with_retrieve(self, request, status=status.HTTP_201_CREATED, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)

        response_serializer = self.get_retrieve_serializer(serializer.instance)
        headers = self.get_success_headers(response_serializer.data)
        return Response(response_serializer.data, status=status, headers=headers)

    def create(self, request, status=status.HTTP_201_CREATED, *args, **kwargs):
        try:
            return self.create_with_retrieve(request, status=status, *args, **kwargs)
        except DjangoValidationError as exc:
            raise ValidationError(detail=serializers.as_serializer_error(exc))


class UpdateWithRetrieveModelMixin:
    """
    Обновление модели с ответом как в get-ручке
    """

    def update_with_retrieve(self, request, *args, **kwargs):
        partial = kwargs.pop('partial', False)
        instance = self.get_object()
        serializer = self.get_serializer(instance, data=request.data, partial=partial)
        serializer.is_valid(raise_exception=True)
        self.perform_update(serializer)

        if getattr(instance, '_prefetched_objects_cache', None):
            # If 'prefetch_related' has been applied to a queryset, we need to
            # forcibly invalidate the prefetch cache on the instance.
            instance._prefetched_objects_cache = {}

        response_serializer = self.get_retrieve_serializer(serializer.instance)
        return Response(response_serializer.data)

    def partial_update_with_retrieve(self, request, *args, **kwargs):
        kwargs['partial'] = True
        return self.update_with_retrieve(request, *args, **kwargs)

    def update(self, request, *args, **kwargs):
        try:
            return self.update_with_retrieve(request, *args, **kwargs)
        except DjangoValidationError as exc:
            raise ValidationError(detail=serializers.as_serializer_error(exc))

    def partial_update(self, request, *args, **kwargs):
        try:
            return self.partial_update_with_retrieve(request, *args, **kwargs)
        except DjangoValidationError as exc:
            raise ValidationError(detail=serializers.as_serializer_error(exc))


class DeleteProtectedModelMixin:
    def perform_destroy(self, instance: models.Model):
        try:
            return super().perform_destroy(instance)
        except DjangoValidationError as exc:
            raise ValidationError(detail=serializers.as_serializer_error(exc))
        except models.deletion.ProtectedError as exc:
            log.warning(exc)
            protected_obj = exc.protected_objects.first()
            obj_title = _('связанные данные') if not protected_obj else protected_obj._meta.verbose_name
            error_msg = _("нельзя удалить, потому что есть %s" % obj_title),
            raise ValidationError(detail=error_msg, code='protected')


class TVMUidMixin:
    def get_tvm_uid(self):
        return getattr(self.request, 'tvm_uid', None)


class IsPreviewMixin:
    @property
    def is_preview(self):
        value = self.request.query_params.get('preview')
        if value:
            return (
                self.action in ['list', 'retrieve'] and
                BooleanField().to_representation(value)
            )

        return False


class ApiAllowedTVMServiceMixin:
    permission_classes = [
        ApiAllowedTVMServicePermission,
    ]


class LabApiAllowedTVMServiceMixin:
    permission_classes = [
        LabApiAllowedTVMServicePermission,
    ]


class LabPermissionsMixin:
    permission_classes = [
        LabPermissions,
    ]
