from django.utils.encoding import force_str
from django.utils.translation import gettext_lazy as _

from rest_framework.compat import coreapi, coreschema
from rest_framework.pagination import LimitOffsetPagination


class LimitOffsetAllPagination(LimitOffsetPagination):
    all_query_param = 'all'
    all_query_description = _('Retrieve all results at once')

    def paginate_queryset(self, queryset, request, view=None):
        show_all = bool(request.query_params.get(self.all_query_param, False))
        if not show_all:
            return super().paginate_queryset(queryset, request, view)

        self.count = self.get_count(queryset)
        self.limit = self.count
        self.offset = self.get_offset(request)
        self.request = request

        if self.count == 0 or self.offset > self.count:
            return []

        return list(queryset[self.offset:self.offset + self.limit])

    def get_schema_fields(self, view):
        fields = super().get_schema_fields(view)
        return fields + [
            coreapi.Field(
                name=self.all_query_param,
                required=False,
                location='query',
                schema=coreschema.Boolean(
                    title='ShowAll',
                    description=force_str(self.all_query_description)
                )
            )
        ]

    def get_schema_operation_parameters(self, view):
        parameters = super().get_schema_operation_parameters(view)
        return parameters + [
            {
                'name': self.all_query_param,
                'required': False,
                'in': 'query',
                'description': force_str(self.all_query_description),
                'schema': {
                    'type': 'boolean',
                }
            }
        ]
