import logging

import jsonschema

from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _

from .rules import get_rule
from .schema import RULES_SCHEMA

logger = logging.getLogger(__name__)


class RuleParseError(ValidationError):
    pass


class RuleTreeParser:
    EQUAL = 'eq'
    NOT_EQUAL = 'ne'
    binary_operators = {
        EQUAL,
        NOT_EQUAL,
    }

    OR = 'or'
    AND = 'and'
    multiple_operators = {
        OR,
        AND,
    }

    def __init__(self, tree: dict):
        self.validate_schema(tree)

        self.parameters = {}
        self.formula = self.build(tree)
        self.check_parameters()

    def validate_schema(self, tree: dict):
        try:
            jsonschema.validate(tree, RULES_SCHEMA)
        except jsonschema.exceptions.ValidationError:
            logger.exception("invalid rules schema")
            raise RuleParseError(_("Ошибка парсинга правил: некорректная схема"), code='invalid_jsonschema')

    def check_parameters(self):
        errors = []
        for name in self.parameters.keys():
            rule_name = name.split('__')[0]
            if not get_rule(rule_name):
                errors.append(rule_name)

        if errors:
            msg = _("Эти правила видимости не найдены или не существуют: %r") % ",".join(errors)
            raise RuleParseError(msg, code='unknown_rule')

    def build(self, tree: dict, is_child=False):
        try:
            op_name, nodes = list(tree.items())[0]
        except Exception:
            raise RuleParseError(_("Ошибка формата дерева правил"), code='wrong_tree_format')

        if op_name not in self.multiple_operators:
            if op_name not in self.binary_operators:
                raise RuleParseError(_("Неизвестный оператор: %s") % op_name, code='unknown_operator')

            if not len(nodes) == 2:
                raise RuleParseError(
                    _("Правило должно иметь следующий формат: [name, values]"),
                    code='rule_wrong_format',
                )

            key, value = nodes
            uniq_key = '{}__{}'.format(key, len(self.parameters) + 1)
            self.parameters[uniq_key] = value
            return '{} is {}'.format(uniq_key, op_name == self.EQUAL)

        elements = [self.build(node, is_child=True) for node in nodes]
        result = " {} ".format(op_name).join(elements)
        if is_child:
            result = f"({result})"

        return result
