from django.contrib.auth import get_user_model

from ..validators import BaseValidator, MatchValidator

User = get_user_model()


class RuleRegistry(type):
    _registry = {}

    def __new__(cls, name, bases, attrs):
        new_cls = super().__new__(cls, name, bases, attrs)

        if not attrs.pop('abstract', None):
            name = getattr(new_cls, 'label', new_cls.__name__.lower())
            cls._registry[name] = new_cls
        return new_cls

    @classmethod
    def get_registry(cls):
        return dict(cls._registry)

    @classmethod
    def get_rule(cls, name: str, default=None):
        return cls._registry.get(name, default)


class BaseRule(metaclass=RuleRegistry):
    label = 'default_rule'
    validator_class = MatchValidator

    abstract = True

    def __init__(self, initial):
        args, kwargs = self.get_validator_params(initial)
        self.validator = self.get_validator(*args, **kwargs)

    def __call__(self, *args, **kwargs):
        return self.check(*args, **kwargs)

    def get_validator(self, *args, **kwargs):
        if not issubclass(self.validator_class, BaseValidator):
            raise TypeError('validator_class must be subclass of `BaseValidator')

        return self.validator_class(*args, **kwargs)

    def get_validator_params(self, initial):
        args = []
        kwargs = {}

        if isinstance(initial, list):
            args = initial

        elif isinstance(initial, dict):
            kwargs.update(initial)

        else:
            args.append(initial)

        return args, kwargs

    def get_user_data(self, user: User):
        raise NotImplementedError("Method get_user_data() must be implemented.")

    def check(self, user: User):
        user_data = self.get_user_data(user)
        if not user_data:
            return False

        return self.validator(user_data)
