import getpass
from contextlib import closing
from uuid import uuid4

import psycopg2
from psycopg2.extras import DictCursor
from tqdm import tqdm

from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from django.db import transaction
from django.utils.text import slugify

from lms.contrib.staff.client import staff_api
from lms.courses.models import (
    ColorTheme, Course, CourseCategory, CourseCity, CourseGroup, CourseStudent, CourseTeam, CourseVisibility, StudyMode,
)
from lms.enrollments.models import EnrolledUser, Enrollment
from lms.tracker.models import EnrollmentTrackerIssue, EnrollmentTrackerQueue

User = get_user_model()


class DryRunException(Exception):
    pass


COLOR_THEME_MAP = {
    'YDS': 'linear-gradient(to right, #ff6757, #ffe266)',
    'expertise': 'linear-gradient(to right, #f95146, #ffbba7)',
    'foreign_languages': 'linear-gradient(to right, #f9ae1a, #fff698)',
    'internal_processes': 'linear-gradient(to right, #55c151, #d1ffb6)',
    'management': 'linear-gradient(to right, #57aaff, #8decca)',
    'online_platforms': 'linear-gradient(to right, #57aaff, #afddff)',
    'personal_efficiency': 'linear-gradient(to right, #b580ff, #dfbeff)',
    'programming_languages': 'linear-gradient(to right, #57aaff, #cfa2ff)',
    'project_management': 'linear-gradient(to right, #ff6757, #c898ff)',
    'other': 'linear-gradient(to right, #40b29f, #b6ffdc)',
}

DEFAULT_COLOR = 'linear-gradient(to right, #ff6757, #ffe266)'

YDS_REGULAR = 'yds_regular'
EXTERNAL_LEARNING = 'external_learning'
EXTERNAL_COACH = 'external_coach'
EXTERNAL_ONLINE = 'extrnal_online'
ONLINE = 'online'
SUBSCRIPTION = 'subscription'
INTERNAL_LEARNING = 'internal_learning'
OTHER = 'other'

FORMAT_AUTHOR_MAP = {
    YDS_REGULAR: 1120000000164934,  # eseni
    EXTERNAL_LEARNING: 1120000000034384,  # mashafedo
    EXTERNAL_COACH: 1120000000034384,  # mashafedo
    EXTERNAL_ONLINE: 1120000000034384,  # mashafedo
    OTHER: 1120000000034384,  # mashafedo
    ONLINE: 1120000000046169,  # armineakh
    SUBSCRIPTION: 1120000000046169,  # armineakh
    INTERNAL_LEARNING: 1120000000041215,  # nikar86
}

FORMAT_TEAM_MAP = {
    EXTERNAL_LEARNING: 'external_team',  # mashafedo
    EXTERNAL_COACH: 'external_team',  # mashafedo
    EXTERNAL_ONLINE: 'external_team',  # mashafedo
    OTHER: 'external_team',  # mashafedo
    ONLINE: 'online_team',  # mashafedo
    SUBSCRIPTION: 'online_team',  # mashafedo
}


class Command(BaseCommand):
    def fill_color_theme_map(self):
        existed_color_themes = ColorTheme.objects.filter(
            course_card_gradient_color__in=list(COLOR_THEME_MAP.values()) + [DEFAULT_COLOR],
        )
        existed_color_theme_map = {
            existed_color_theme.course_card_gradient_color: existed_color_theme
            for existed_color_theme in existed_color_themes
        }

        color_themes_to_create = []
        for theme_name, color in COLOR_THEME_MAP.items():
            if color in existed_color_theme_map:
                color_theme = existed_color_theme_map[color]
            else:
                color_theme = ColorTheme(
                    slug=slugify(theme_name), name=theme_name, is_active=True, course_card_gradient_color=color,
                )
                color_themes_to_create.append(color_theme)
            self.color_theme_map[theme_name] = color_theme
        if DEFAULT_COLOR in existed_color_theme_map:
            self.default_color_theme = existed_color_theme_map[DEFAULT_COLOR]
        else:
            self.default_color_theme = ColorTheme(
                slug=slugify('default_color_theme'),
                name='default_color_theme',
                is_active=True,
                course_card_gradient_color=DEFAULT_COLOR,
            )
            color_themes_to_create.append(self.default_color_theme)
        ColorTheme.objects.bulk_create(color_themes_to_create)

    def get_color_theme(self, theme):
        return self.color_theme_map.get(theme, self.default_color_theme)

    def get_or_create_user(self, uid=None, login=None):
        filters = {'_fields': 'uid,name.last.ru,name.first.ru,login'}
        if uid is not None:
            if uid in self.user_map_uid:
                return self.user_map_uid[uid]
            filters['uid'] = uid
        elif login is not None:
            if login in self.user_map_login:
                return self.user_map_login[login]
            filters['login'] = login
        else:
            return None

        res = getattr(staff_api, 'persons').get(**filters)
        uid = res["result"][0]["uid"]
        login = res["result"][0]["login"]
        first_name = res["result"][0]["name"]["first"]["ru"]
        last_name = res["result"][0]["name"]["last"]["ru"]

        if uid in self.user_map_uid:
            user = self.user_map_uid[uid]
        elif login in self.user_map_login:
            user = self.user_map_login[login]
        else:
            user, _ = User.objects.get_or_create(
                yauid=uid,
                defaults={
                    'username': login,
                    'first_name': first_name,
                    'last_name': last_name,
                }
            )
            self.user_map_uid[uid] = user
            self.user_map_login[login] = user

        return user

    def import_courses(self, conn):
        self.stdout.write('Import courses')

        # Запрос курсов
        course_query = """
        SELECT
            *,
            education_program.id AS program_id,
            auth_user.uid AS user_uid,
            auth_user.username AS user_username
        FROM
            education_program
            LEFT JOIN auth_user ON education_program.head_id = auth_user.id
        """
        # course_query = """
        # SELECT
        #     *
        # FROM
        #     education_program
        # """

        if self.courses_ids:
            course_query += f'WHERE education_program.id IN ({", ".join([str(i) for i in self.courses_ids])})'  # nosec
        course_count_query = f'SELECT COUNT(*) AS _count FROM ({course_query}) AS _t1'  # nosec

        # Считаем курсы
        count = 0
        with conn.cursor(cursor_factory=DictCursor) as cursor:
            cursor.execute(course_count_query)
            for row in cursor:
                count = row['_count']
                break
        with conn.cursor(cursor_factory=DictCursor) as cursor:
            cursor.execute(course_query)

            theme_parent_category, _ = CourseCategory.objects.get_or_create(name='Темы')
            study_mode = StudyMode.objects.get(id=self.study_mode_id)
            default_author = User.objects.get(id=self.default_author_id)
            all_courses_team = CourseTeam.objects.get(id=self.team_id)

            external_team = CourseTeam.objects.get(id=self.external_team_id)
            FORMAT_TEAM_MAP[EXTERNAL_LEARNING] = external_team
            FORMAT_TEAM_MAP[EXTERNAL_COACH] = external_team
            FORMAT_TEAM_MAP[EXTERNAL_ONLINE] = external_team
            FORMAT_TEAM_MAP[OTHER] = external_team

            online_team = CourseTeam.objects.get(id=self.online_team_id)
            FORMAT_TEAM_MAP[ONLINE] = online_team
            FORMAT_TEAM_MAP[SUBSCRIPTION] = online_team

            for format_item in FORMAT_AUTHOR_MAP:
                FORMAT_AUTHOR_MAP[format_item] = self.get_or_create_user(uid=FORMAT_AUTHOR_MAP[format_item])

            for row in tqdm(cursor, total=count):
                program_format = row['format']
                city_name = row['city']
                if city_name:
                    city_name = city_name.strip().title()
                    if city_name in self.city_map:
                        city = self.city_map[city_name]
                    else:
                        city, _ = CourseCity.objects.get_or_create(
                            name=city_name,
                            defaults={
                                'slug': str(uuid4()),
                            },
                        )
                        self.city_map[city_name] = city
                else:
                    city = None

                uid = row['user_uid']
                login = row['user_username']
                author = (
                    self.get_or_create_user(uid=uid, login=login) or
                    FORMAT_AUTHOR_MAP.get(program_format, None) or
                    default_author
                )

                theme_name = row['theme']
                color_theme = self.get_color_theme(theme_name)
                if theme_name in self.category_map:
                    theme = self.category_map[theme_name]
                else:
                    theme, _ = CourseCategory.objects.get_or_create(
                        name=theme_name,
                        parent=theme_parent_category,
                        defaults={
                            'slug': str(uuid4()),
                        }
                    )
                    self.category_map[theme_name] = theme

                course_name = (row['name'] or f"Курс {row['program_id']}")[:255]
                course_slug = slugify(f"{row['name']}-{row['program_id']}")[:255]
                if self.new_only:
                    method = 'get_or_create'
                else:
                    method = 'update_or_create'
                course, created = getattr(Course.objects, method)(
                    hrdb_id=row['program_id'],
                    defaults={
                        'slug': course_slug,
                        'price': row['cost_per_person'],
                        'paid_percent': row['retention_percent'] or 0,
                        'created': row['created_at'],
                        'modified': row['updated_at'],
                        'city': city,
                        'study_mode': study_mode,
                        'color_theme': color_theme,
                        'author': author,
                        'name': course_name,
                        'summary': (row['details'] or '')[:500],
                        'description':
                            f"{row['description'] if row['description'] else ''}\n"
                            f"Ссылка: {row['link']}\n"
                            f"Продолжительность: {row['duration']}\n"
                            f"Количество уроков% {row['lessons']}\n",
                        'enroll_begin': row['subscription_date_from'],
                        'enroll_end': row['subscription_date_to'],
                    },
                )
                if created:
                    self.created_courses += 1
                    self.created_courses_ids.append(row['program_id'])
                elif not self.new_only:
                    self.updated_courses += 1
                    self.updated_courses_ids.append(row['program_id'])

                self.course_map[course.hrdb_id] = course
                course.categories.add(theme)
                course.teams.add(all_courses_team)
                if program_format in FORMAT_TEAM_MAP:
                    course.teams.add(FORMAT_TEAM_MAP[program_format])

                if row['only_for_heads']:
                    CourseVisibility.objects.update_or_create(
                        course_id=course.id,
                        defaults={
                            'rules': {"eq": ["staff_is_head", True]}
                        }
                    )

                enrollment, _ = Enrollment.objects.update_or_create(
                    course=course,
                    enroll_type=Enrollment.TYPE_TRACKER,
                    defaults={
                        'is_active': True,
                        'name': "Зачисление через трекер",
                    }
                )

                for queue, is_default in zip(
                    (row['startrek_main_queue'], row['startrek_agreement_queue']),
                    (True, False),
                ):
                    if queue:
                        queue, created = EnrollmentTrackerQueue.objects.update_or_create(
                            enrollment=enrollment,
                            name=queue,
                            defaults={
                                'issue_type': row['startrek_main_type'],
                                'is_default': is_default,
                            }
                        )
                        if created:
                            self.created_queues += 1
                        else:
                            self.updated_queues += 1

    def import_groups(self, conn):
        self.stdout.write('Import groups')

        course_group_query = """
        SELECT
            *
        FROM
            education_group
        """

        if self.courses_ids:
            ids = ", ".join([str(i) for i in self.courses_ids])
            course_group_query += f'WHERE education_group.program_id IN ({ids})'  # nosec
        course_group_count_query = f'SELECT COUNT(*) AS _count FROM ({course_group_query}) AS _t1'  # nosec
        count = 0
        with conn.cursor(cursor_factory=DictCursor) as cursor:
            cursor.execute(course_group_count_query)
            for row in cursor:
                count = row['_count']
                break
        with conn.cursor(cursor_factory=DictCursor) as cursor:
            cursor.execute(course_group_query)

            for row in tqdm(cursor, total=count):
                course_id = row['program_id']
                if course_id:
                    if course_id in self.course_map:
                        course = self.course_map[course_id]
                    else:
                        course = Course.objects.get(hrdb_id=course_id)
                        self.course_map[course_id] = course
                else:
                    course = None

                if self.new_only:
                    method = 'get_or_create'
                else:
                    method = 'update_or_create'
                course_group, created = getattr(CourseGroup.objects, method)(
                    hrdb_id=row['id'],
                    defaults={
                        'course': course,
                        'name': (row['name'] or f"Курс {course.id}, Группа {row['id']}")[:255],
                        'summary': row['description'] or '',
                        'is_active': row['is_active'],
                        'max_participants': max(row['max_count'] or 0, 0),
                        'created': row['created_at'],
                        'modified': row['updated_at'],
                    },
                )
                if created:
                    self.created_groups += 1
                    self.created_groups_ids.append(row['id'])
                elif not self.new_only:
                    self.updated_groups += 1
                    self.updated_groups_ids.append(row['id'])
                self.group_map[course_group.hrdb_id] = course_group

    def import_enrolled_users(self, conn):
        self.stdout.write('Import enrolled users')

        enrolled_user_query = """
        SELECT
            *,
            education_enlistment.id AS enlistment_id,
            education_enlistment.employee_id AS employee_id,
            t_employee.uid AS employee_uid,
            t_employee.username AS employee_username,
            education_enlistment.user_id AS user_id,
            t_user.uid AS user_uid,
            t_user.username AS user_username
        FROM
            education_enlistment
            LEFT JOIN auth_user AS t_employee ON education_enlistment.employee_id = t_employee.id
            LEFT JOIN auth_user AS t_user ON education_enlistment.user_id = t_user.id
        """
        # enrolled_user_query = """
        # SELECT
        #     *
        # FROM
        #     education_enlistment
        # """

        if self.courses_ids:
            ids = ", ".join([str(i) for i in self.courses_ids])
            enrolled_user_query += f'WHERE education_enlistment.program_id IN ({ids})'  # nosec
        enrolled_user_count_query = f'SELECT COUNT(*) AS _count FROM ({enrolled_user_query}) AS _t1'  # nosec
        count = 0
        with conn.cursor(cursor_factory=DictCursor) as cursor:
            cursor.execute(enrolled_user_count_query)
            for row in cursor:
                count = row['_count']
                break
        with conn.cursor(cursor_factory=DictCursor) as cursor:
            cursor.execute(enrolled_user_query)

            for row in tqdm(cursor, total=count):
                group_id = row['group_id']
                if group_id:
                    if group_id in self.group_map:
                        group = self.group_map[group_id]
                    else:
                        group = CourseGroup.objects.get(hrdb_id=group_id)
                        self.group_map[group_id] = group
                else:
                    group = None

                course_id = row['program_id']
                if course_id:
                    if course_id in self.course_map:
                        course = self.course_map[course_id]
                    else:
                        course = Course.objects.get(hrdb_id=course_id)
                        self.course_map[course_id] = course
                elif group is not None:
                    course = group.course
                else:
                    continue

                user_id = row['user_id']
                employee_id = row['employee_id']
                if employee_id is not None:
                    uid = row['employee_uid']
                    username = row['employee_username']
                elif user_id is not None:
                    uid = row['user_uid']
                    username = row['user_username']
                else:
                    continue
                user = self.get_or_create_user(uid=uid, login=username)

                enrollment = Enrollment.objects.get(
                    course=course,
                    enroll_type=Enrollment.TYPE_TRACKER,
                )
                enrolled_user = EnrolledUser.objects.filter(
                    course=course,
                    user=user,
                    group=group,
                ).first()
                if enrolled_user is None:
                    if self.new_only:
                        method = 'get_or_create'
                    else:
                        method = 'update_or_create'
                    enrolled_user, created = getattr(EnrolledUser.objects, method)(
                        hrdb_id=row['enlistment_id'],
                        defaults={
                            'course': course,
                            'user': user,
                            'group': group,
                            'enrollment': enrollment,
                            'status':
                                self.enrolled_user_status_map.get(row['status'], EnrolledUser.StatusChoices.ENROLLED),
                            'hrdb_survey_data': row['survey_answers'] or '{}',
                            'created': row['created_at'],
                        },
                    )
                    if created:
                        self.created_enrolled_users += 1
                    elif not self.new_only:
                        self.updated_enrolled_users += 1

                course_student_status = self.course_student_status_map.get(
                    row['status'], CourseStudent.StatusChoices.ACTIVE,
                )
                if course_student_status == CourseStudent.StatusChoices.ACTIVE:
                    CourseStudent.objects.update_or_create(
                        course=course,
                        user=user,
                        group=group,
                        defaults={
                            'status': course_student_status,
                        }
                    )

                for ticket in (row['ticket'], row['extra_ticket']):
                    if ticket:
                        queue_name, issue_number = ticket.split('-')
                        try:
                            issue_number = int(issue_number)
                        except ValueError:
                            continue
                        queue, created = EnrollmentTrackerQueue.objects.get_or_create(
                            enrollment=enrollment,
                            name=queue_name,
                            defaults={
                                'is_default': False,
                            },
                        )
                        if created:
                            self.created_queues += 1

                        issue, created = EnrollmentTrackerIssue.objects.update_or_create(
                            enrolled_user=enrolled_user,
                            queue=queue,
                            defaults={
                                'got_status_from_startrek': row['created_at'],
                                'status_processed': True,
                                'issue_number': issue_number,
                            }
                        )
                        if created:
                            self.created_issues += 1
                        else:
                            self.updated_issues += 1

    def import_hrdb(self, options):

        self.city_map = {}
        self.category_map = {}
        self.user_map_uid = {}
        self.user_map_login = {}
        self.course_map = {}
        self.group_map = {}
        self.color_theme_map = {}
        self.default_color_theme = None
        self.fill_color_theme_map()
        self.courses_ids = options.get('courses_ids')
        self.study_mode_id = options.get('study_mode_id')
        self.default_author_id = options.get('default_author_id')
        self.team_id = options.get('team_id')
        self.external_team_id = options.get('external_team_id')
        self.online_team_id = options.get('online_team_id')
        self.new_only = options.get('new_only', False)

        self.created_courses = 0
        self.created_courses_ids = []
        self.updated_courses = 0
        self.updated_courses_ids = []
        self.created_groups = 0
        self.created_groups_ids = []
        self.updated_groups = 0
        self.updated_groups_ids = []
        self.created_enrolled_users = 0
        self.updated_enrolled_users = 0
        self.created_queues = 0
        self.updated_queues = 0
        self.created_issues = 0
        self.updated_issues = 0

        self.enrolled_user_status_map = {
            'new': EnrolledUser.StatusChoices.PENDING,
            'active': EnrolledUser.StatusChoices.ENROLLED,
            'canceled': EnrolledUser.StatusChoices.REJECTED,
        }
        self.course_student_status_map = {
            'active': CourseStudent.StatusChoices.ACTIVE,
        }

        with closing(
            psycopg2.connect(
                dbname=options['hrdb_name'],
                user=options['hrdb_login'],
                password=options['hrdb_password'],
                host=options['hrdb_host'],
                port=options['hrdb_port'],
            ),
        ) as conn:
            try:
                with transaction.atomic():
                    if options['import_mode'] in ['all', 'course']:
                        self.import_courses(conn)
                    if options['import_mode'] in ['all', 'group']:
                        self.import_groups(conn)
                    if options['import_mode'] in ['all', 'enlistment']:
                        self.import_enrolled_users(conn)

                    if options.get('dry_run', False):
                        raise DryRunException

            except DryRunException:
                self.stdout.write('Rollback')
            else:
                self.stdout.write('Commit')

            self.stdout.write(
                f"""
                    courses: created: {self.created_courses}, updated: {self.updated_courses}
                    new courses: {self.created_courses_ids}, updated courses {self.updated_courses_ids}
                    groups: created: {self.created_groups}, updated: {self.updated_groups}
                    new groups: {self.created_groups_ids}, updated groups: {self.updated_groups_ids}
                    enrolled users: created: {self.created_enrolled_users}, updated {self.updated_enrolled_users}
                    queues: created: {self.created_queues}, updated: {self.updated_queues}
                    issues: created: {self.created_issues}, updated: {self.updated_issues}
                """
            )

    def add_arguments(self, parser):
        parser.add_argument(
            '--host', dest='hrdb_host', type=str, required=True,
            help='HRDB host',
        )
        parser.add_argument(
            '--name', '-n', dest='hrdb_name', type=str, required=True,
            help='HRDB name',
        )
        parser.add_argument(
            '--login', '-l', dest='hrdb_login', type=str, required=True,
            help='HRDB login',
        )
        parser.add_argument(
            '--password', '-w', dest='hrdb_password', type=str,
            help='HRDB password (stay empty to enter in protection mode)',
        )
        parser.add_argument(
            '--port', '-p', dest='hrdb_port', type=int, default=5432,
            help='HRDB port',
        )
        parser.add_argument(
            '--study-mode-id', '-m', dest='study_mode_id', type=int, required=True,
            help='Study mode id',
        )
        parser.add_argument(
            '--default-author-id', '-a', dest='default_author_id', type=int, required=True,
            help='Default author id',
        )
        parser.add_argument(
            '--dry-run', action='store_true', dest='dry_run',
            help='Run without commit to db',
        )
        parser.add_argument(
            '--new-only', action='store_true', dest='new_only',
            help='Run without commit to db',
        )
        parser.add_argument(
            '-c', '--courses', dest='courses_ids', type=int, required=False, nargs='*',
            help='Courses ids',
        )
        parser.add_argument(
            '--mode', dest='import_mode', type=str, default='all',
            choices=['course', 'group', 'enlistment', 'all'], help='Import mode',
        )
        parser.add_argument(
            '--team-id', dest='team_id', type=int,
            help='Team for all courses',
        )
        parser.add_argument(
            '--external-team-id', dest='external_team_id', type=int,
            help='Team for external courses',
        )
        parser.add_argument(
            '--online-team-id', dest='online_team_id', type=int,
            help='Team for online courses',
        )

    def handle(self, *args, **options):
        if options.get('hrdb_password') is None:
            options['hrdb_password'] = getpass.getpass()

        self.import_hrdb(options)

        self.stdout.write("Done\n")
