import logging
from datetime import datetime
from typing import Optional

from django_ltree.models import TreeModel
from model_utils import FieldTracker
from model_utils.models import TimeStampedModel
from ordered_model.models import OrderedModel
from simple_history.models import HistoricalRecords

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.hashers import UnsaltedSHA1PasswordHasher
from django.contrib.auth.models import Group
from django.contrib.postgres.fields import JSONField
from django.core.exceptions import ValidationError
from django.core.validators import MaxValueValidator, MinValueValidator
from django.db import models
from django.db.models import Min, Q
from django.utils import timezone
from django.utils.functional import cached_property
from django.utils.translation import gettext_lazy as _

from lms.contrib.s3upload.fields import S3UploadField
from lms.core.models.mixins import ActiveModelMixin, HrdbIdModelMixin
from lms.core.validators import list_of_logins_validator
from lms.core.visibility.parser import RuleParseError, RuleTreeParser
from lms.core.visibility.services import available_for_user
from lms.moduletypes.models import Module as BaseModule
from lms.preferences.models import ColorTheme
from lms.tags.models import Tag
from lms.users.models import PermissionPreset, ServiceAccount

from .managers import CourseGroupManager, CourseVisibilityManager
from .mixins import AvailabilityMixin
from .querysets import (
    CohortQuerySet, CourseBlockQuerySet, CourseCategoryQuerySet, CourseCityQuerySet, CourseGroupQuerySet,
    CourseModuleQuerySet, CourseQuerySet, CourseStudentQuerySet, CourseWorkflowQuerySet, ProviderQuerySet,
    StudyModeQuerySet, TutorQuerySet,
)
from .utils import files_upload_destination
from .validators import validate_course_slug

logger = logging.getLogger(__name__)

User = get_user_model()


class Tutor(TimeStampedModel):
    class PositionChoices(models.TextChoices):
        COACH = 'coach', _("тренер")
        TEACHER = 'teacher', _("преподаватель")

    is_internal = models.BooleanField(_("внутренний"))
    user = models.ForeignKey(
        User,
        verbose_name=_("внутренний наставник"),
        null=True,
        blank=True,
        on_delete=models.PROTECT,
    )
    name = models.CharField(
        max_length=1024,
        verbose_name=_("имя внешнего наставника"),
        blank=True,
    )
    email = models.EmailField(_("email внешнего наставника"), blank=True)
    url = models.CharField(
        max_length=1024,
        verbose_name=_("ссылка по профиль внешнего наставника"),
        blank=True,
    )
    is_active = models.BooleanField(_("активен"), default=True)
    position = models.CharField(max_length=10, verbose_name=_("должность"), choices=PositionChoices.choices)

    objects = TutorQuerySet.as_manager()

    @property
    def name_common(self) -> str:
        return f'{self.user.first_name} {self.user.last_name}' if self.is_internal else self.name

    @property
    def url_common(self) -> str:
        if self.is_internal:
            return self.user.staffprofile.staff_profile_url if hasattr(self.user, 'staffprofile') else ""

        return self.url

    class Meta:
        verbose_name = _("наставник")
        verbose_name_plural = _("наставники")

    def __str__(self):
        return str(self.user) if self.is_internal else self.name

    def clean(self):
        if self.is_internal and not self.user:
            raise ValidationError(
                {'user': ValidationError(_("для внутреннго наставника не указан пользователь"), code='required')}
            )

        if not self.is_internal and not self.name:
            raise ValidationError({'name': ValidationError(
                _("для внешнего наставника нужно указать хотя бы одно из полей"),
                code='required',
            )})

    def save(self, *args, **kwargs):
        self.full_clean()
        return super().save(*args, **kwargs)


class StudyMode(OrderedModel, TimeStampedModel, ActiveModelMixin):
    slug = models.SlugField(_("код"), max_length=255, blank=True, unique=True)
    name = models.CharField(_("название"), max_length=255)
    description = models.TextField(_("описание"), blank=True)

    objects = StudyModeQuerySet.as_manager()

    class Meta:
        ordering = ('order',)
        verbose_name = _("форма обучения")
        verbose_name_plural = _("формы обучения")

    def __str__(self):
        return f'{self.name} [{self.slug}]' if self.slug else self.name


class Provider(TimeStampedModel, ActiveModelMixin):
    oebs_id = models.CharField(_("ID в OEBS"), max_length=255, blank=True)
    name = models.CharField(_("название"), max_length=255)
    description = models.TextField(_("описание"), blank=True)

    created_by = models.ForeignKey(
        User,
        verbose_name=_("создано"),
        related_name='created_providers',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    objects = ProviderQuerySet.as_manager()

    class Meta:
        ordering = ('name',)
        verbose_name = _("провайдер")
        verbose_name_plural = _("провайдеры")
        permissions = (
            ('import_provider', _('Can export providers')),
            ('export_provider', _('Can import providers')),
        )

    def __str__(self):
        return self.name


class CourseCity(TimeStampedModel):
    name = models.CharField(_("название"), max_length=255)
    slug = models.SlugField(_("код"), max_length=255, unique=True)
    is_active = models.BooleanField(_("активен"), default=True)

    objects = CourseCityQuerySet.as_manager()

    class Meta:
        ordering = ('name',)
        verbose_name = _("город")
        verbose_name_plural = _("города")

    def __str__(self):
        return self.name


class CourseCategory(TimeStampedModel):
    parent = models.ForeignKey(
        'self',
        verbose_name=_("Родительская категория"),
        related_name='children',
        null=True,
        blank=True,
        db_index=True,
        on_delete=models.PROTECT,
    )

    slug = models.SlugField(_("код"), max_length=255, unique=True)
    name = models.CharField(_("название"), max_length=255)
    description = models.TextField(_("описание"), blank=True)
    color_theme = models.ForeignKey(
        ColorTheme,
        verbose_name=_("цветовая схема"),
        related_name='categories',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    is_active = models.BooleanField(_("активен"), default=True)
    created_by = models.ForeignKey(
        User,
        verbose_name=_("создано"),
        related_name='created_course_categories',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    objects = CourseCategoryQuerySet.as_manager()

    class Meta:
        ordering = ('name',)
        verbose_name = _("категория курса")
        verbose_name_plural = _("категории курсов")

    def __str__(self):
        return self.name

    @cached_property
    def node_depth(self):
        return self.node.depth

    @property
    def with_courses(self) -> bool:
        return self.courses.exists()


class CategoryNode(TreeModel):
    category = models.OneToOneField(
        CourseCategory,
        related_name='node',
        verbose_name=_("категория"),
        primary_key=True,
        on_delete=models.CASCADE,
    )

    class Meta:
        ordering = ("path",)
        verbose_name = _("узел категории")
        verbose_name_plural = _("дерево категорий")

    def __str__(self):
        return '{0} [{1}]'.format(self.category, self.path)

    @property
    def depth(self):
        return len(self.path)


class CourseSettings(models.Model):
    class PaymentMethodChoices(models.TextChoices):
        FREE = 'free', _("Бесплатно")
        CORPORATE = 'corporate', _("За счет компании")
        PERSONAL = 'personal', _("С удержанием")

    begin_date = models.DateTimeField(_("начало обучения"), null=True, blank=True, db_index=True)
    end_date = models.DateTimeField(_("окончание обучения"), null=True, blank=True, db_index=True)

    enroll_begin = models.DateTimeField(_("начало зачисления"), null=True, blank=True, db_index=True)
    enroll_end = models.DateTimeField(_("окончание зачисления"), null=True, blank=True, db_index=True)

    price = models.DecimalField(
        _("стоимость в каталоге"), max_digits=10, decimal_places=2,
        null=True, blank=True,
        help_text=_("Полная стоимость, запрашиваемая провайдером"),
    )

    payment_method = models.CharField(
        _("вариант оплаты"), max_length=20,
        choices=PaymentMethodChoices.choices,
        default=PaymentMethodChoices.FREE,
    )

    paid_percent = models.IntegerField(
        _("процент удержания"), default=0,
        help_text=_("Актуален, только если платит сотрудник"),
    )
    payment_terms = models.TextField(_(
        "условия оплаты"), blank=True,
        help_text=_("Описание условий оплаты, удержаний, стоимости для компании"),
    )
    num_hours = models.IntegerField(
        _("кол-во часов"), null=True, blank=True,
        help_text=_("Количество неакадемических часов"),
    )

    @property
    def is_enroll_open(self) -> bool:
        raise NotImplementedError()

    @property
    def is_full(self) -> bool:
        raise NotImplementedError()

    @property
    def has_open_seats(self) -> bool:
        return not self.is_full

    class Meta:
        abstract = True

    def check_enroll_open(self, now=None) -> bool:
        """
        Проверяет открытие регистрации на момент времени

        :param now:
        :return:
        """
        if now is None:
            now = timezone.now().replace(second=0, microsecond=0)

        if self.enroll_begin is None and self.enroll_end is None:
            return True
        if self.enroll_begin is None:
            return now <= self.enroll_end
        if self.enroll_end is None:
            return self.enroll_begin <= now

        return self.enroll_begin <= now <= self.enroll_end


class CourseTeam(Group):
    permission_preset = models.ForeignKey(
        to=PermissionPreset,
        verbose_name=_("набор разрешений"),
        related_name='teams',
        null=True,
        blank=True,
        on_delete=models.PROTECT,
    )

    @property
    def members(self):
        return self.user_set.all()

    def add_user(self, user_id: int) -> None:
        self.user_set.add(user_id)

    def remove_user(self, user_id: int) -> None:
        self.user_set.remove(user_id)

    def can_delete(self) -> bool:
        return not self.user_set.exists()

    class Meta:
        ordering = ('name',)
        verbose_name = _("Команда курса")
        verbose_name_plural = _("Команды курсов")


class CourseTeamForSupport(CourseTeam):
    class Meta:
        proxy = True
        verbose_name = _("[Для поддержки] Команда курса")
        verbose_name_plural = _("[Для поддержки] Команды курсов")


class CourseWorkflow(TimeStampedModel, ActiveModelMixin):
    name = models.CharField(_("название"), max_length=255, unique=True)

    objects = CourseWorkflowQuerySet.as_manager()

    class Meta:
        ordering = ('name',)
        verbose_name = _("Воркфлоу")
        verbose_name_plural = _("Воркфлоу")

    def __str__(self):
        return self.name


class Course(AvailabilityMixin, CourseSettings, TimeStampedModel, HrdbIdModelMixin):
    class StructureChoices(models.TextChoices):
        NO_MODULES = 'no_modules', _("Без модулей")
        SINGLE = 'single_module', _("Один модуль")
        MULTI = 'multi_modules', _("Несколько модулей")

    class FormatChoices(models.TextChoices):
        SELF_STUDY = 'self_study', _('Самостоятельный')
        WITH_TEACHER = 'with_teacher', _('С преподавателем')

    class TypeChoices(models.TextChoices):
        COURSE = 'course', _('Курс')
        TRACK = 'track', _('Программа')

    begin_date_field = 'calc_begin_date'
    end_date_field = 'calc_end_date'
    enroll_begin_field = 'calc_enroll_begin'
    enroll_end_field = 'calc_enroll_end'

    workflow = models.ForeignKey(
        to=CourseWorkflow,
        verbose_name=_("воркфлоу"),
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    categories = models.ManyToManyField(
        CourseCategory,
        verbose_name=_("категории"),
        related_name='courses',
        blank=True,
    )
    city = models.ForeignKey(
        CourseCity,
        verbose_name=_("город"),
        related_name='courses',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    study_mode = models.ForeignKey(
        StudyMode,
        verbose_name=_("форма обучения"),
        related_name='courses',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    provider = models.ForeignKey(
        Provider,
        verbose_name=_("провайдер"),
        related_name='courses',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    author = models.ForeignKey(
        User,
        verbose_name=_("автор"),
        related_name='courses',
        on_delete=models.PROTECT,
    )
    teams = models.ManyToManyField(
        CourseTeam,
        verbose_name=_("команды курса"),
        related_name='courses',
        blank=True,
    )
    enable_groups = models.BooleanField(
        verbose_name=_("Наличие групп студентов"),
        default=False,
        editable=False,
    )
    # TODO: удалить в пользу указания конкретных подписок LMSDEV-1211
    enable_followers = models.BooleanField(
        verbose_name=_("разрешить подписки на курс"),
        default=False,
        help_text=_("разрешены все подписки на курс"),
    )

    slug = models.SlugField(
        _("код"),
        max_length=255,
        unique=True,
        validators=[validate_course_slug],
    )
    name = models.CharField(_("название"), max_length=255)
    shortname = models.CharField(_("краткое название"), max_length=255, blank=True)
    summary = models.CharField(_("аннотация"), max_length=500, blank=True)
    description = models.TextField(_("описание"), blank=True)
    image_url = models.URLField(_("обложка"), max_length=500, blank=True)

    is_active = models.BooleanField(_("активен"), default=False)
    is_archive = models.BooleanField(_("в архиве"), default=False)

    calc_begin_date = models.DateTimeField(_("фактическое начало обучения"), null=True, blank=True, editable=False)
    calc_end_date = models.DateTimeField(_("фактическое окончание обучения"), null=True, blank=True, editable=False)

    calc_enroll_begin = models.DateTimeField(_("фактическое начало зачисления"), null=True, blank=True, editable=False)
    calc_enroll_end = models.DateTimeField(_("фактическое окончание зачисления"), null=True, blank=True, editable=False)

    structure = models.CharField(
        _("структура курса"),
        max_length=20,
        choices=StructureChoices.choices,
        default=StructureChoices.NO_MODULES,
    )

    format = models.CharField(
        _("формат обучения"),
        max_length=20,
        choices=FormatChoices.choices,
        blank=True,
    )

    enrollments_only = models.BooleanField(
        _("только заявки"),
        default=False,
        help_text=_("""
            При записи на курс создаются только заявки.
            Записи о студентах не создаются.
            Прохождение курса не отслеживается
        """),
    )
    multi_enrollments = models.BooleanField(
        _("множественные заявки"),
        default=False,
        help_text=_("Пользователь может подать на курс несколько заявок"),
    )
    retries_allowed = models.BooleanField(
        _("повторное прохождение"),
        default=False,
        help_text=_("После завершения курса пользователь может снова записаться на курс и пройти его"),
    )

    show_in_catalog = models.BooleanField(
        _("показать в каталоге"),
        default=True,
        help_text=_("если не указано, то курс будет доступен только по прямой ссылке"),
    )

    completion_threshold = models.PositiveSmallIntegerField(
        _("порог завершения"),
        default=100,
        validators=[MaxValueValidator(100)],
        help_text=_("Процент прохождения, необходимый для завершения курса")
    )

    tags = models.ManyToManyField(
        Tag,
        verbose_name=_('теги'),
        related_name='courses',
        blank=True,
        null=True,
    )

    objects = CourseQuerySet.as_manager()

    tracker = FieldTracker(fields=['enable_followers', 'enroll_begin'])

    history = HistoricalRecords()

    course_type = models.CharField(
        _("тип"),
        max_length=20,
        choices=TypeChoices.choices,
        default=TypeChoices.COURSE,
    )

    def get_maximum_capacity(self):
        occ = getattr(self, 'occupancy', None)
        return occ.maximum if occ else None

    def get_current_capacity(self):
        occ = getattr(self, 'occupancy', None)
        return occ.current if occ else None

    @property
    def is_full(self) -> bool:
        """
        Флаг: есть ли свободные места на курсе

        При наличии групп, проверяет места в группах, открытых на регистрацию

        :return:
        """
        # если курс без групп, то места не ограничены
        if not self.enable_groups:
            return False

        for group in self.opened_groups:  # type: CourseGroup
            if group.max_participants == 0 or group.num_participants < group.max_participants:
                return False

        return True

    @property
    def frontend_url(self) -> str:
        return f'{settings.FRONTEND_ROOT}/courses/{self.slug}'

    @property
    def frontend_lab_url(self) -> str:
        return f'{settings.FRONTEND_LAB_ROOT}/courses/{self.slug}'

    @property
    def admin_url(self) -> str:
        return f'{settings.ADMIN_ROOT}/courses/course/{self.id}'

    @property
    def opened_groups(self, now=None):
        """
        Возвращает группы курса, открытые на регистрацию

        Проверяет наличие `_opened_groups`, так можно сделать prefetch по группам
        :param now:
        :return:
        """
        return getattr(self, '_opened_groups', self.groups.opened(now))

    @property
    def is_enroll_open(self) -> bool:
        """
        Флаг: открыта регистрация на курс

        При наличии групп, проверяет открыта ли регистрация хотя бы в одной группе.
        :return:
        """
        if not self.is_active:
            return False

        now = timezone.now().replace(second=0, microsecond=0)

        # если курс с группами, проверяем
        # открыта ли регистрация хотя бы в одной группе
        if self.enable_groups:
            return self.opened_groups.exists()
        else:
            return self.check_enroll_open(now)

    @cached_property
    def enroll_will_begin(self) -> Optional[datetime]:
        """
        Флаг: есть ли предстоящие регистрации

        :return:
        """
        if not self.is_active:
            return None

        now = timezone.now().replace(second=0, microsecond=0)

        if self.enable_groups:
            aggr = self.groups.available().aggregate(
                nearest_enroll_begin=Min('enroll_begin', filter=Q(enroll_begin__gte=now)),
            )
            enroll_begin = aggr.get('nearest_enroll_begin')
        else:
            enroll_begin = self.enroll_begin

        if enroll_begin is not None and enroll_begin > now:
            return enroll_begin

        return None

    class Meta:
        ordering = ('name',)
        verbose_name = _("курс")
        verbose_name_plural = _("курсы")
        permissions = (
            ('import_course', _('Can import courses')),
            ('export_course', _('Can export courses')),
        )

    def clean(self):
        if not self.enrollments_only and self.multi_enrollments:
            raise ValidationError({
                'multi_enrollments': ValidationError(
                    _("настройка 'множественные заявки' возможна только при включенной настройке 'только заявки'"),
                    code='invalid',
                ),
            })

        if self.enrollments_only and self.retries_allowed:
            raise ValidationError({
                'retries_allowed': ValidationError(
                    _("настройка 'повторное прохождение' возможна только при выключенной настройке 'только заявки'"),
                    code='invalid',
                ),
            })

    def save(self, *args, **kwargs):
        if self.course_type == self.TypeChoices.TRACK:
            self.structure = self.StructureChoices.MULTI
        self.full_clean()
        return super().save(*args, **kwargs)

    def __str__(self):
        return f'{self.name} [{self.slug}]' if self.slug else self.name

    def get_group_or_course_field(self, field_name):
        return getattr(self, field_name, None)


class CourseForSupport(Course):
    class Meta:
        proxy = True
        ordering = ('name',)
        verbose_name = _("[Для поддержки] Курс")
        verbose_name_plural = _("[Для поддержки] Курсы")


class CourseOccupancy(TimeStampedModel):
    course = models.OneToOneField(
        Course,
        verbose_name=_("курс"),
        related_name='occupancy',
        on_delete=models.CASCADE,
    )
    current = models.PositiveIntegerField(_("текущая"), default=0)
    maximum = models.PositiveIntegerField(_("максимум"), default=0)

    class Meta:
        verbose_name = _("наполняемость курса")
        verbose_name_plural = _("наполняемость курса")

    def __str__(self):
        return str(self.course_id)


class CourseGroup(AvailabilityMixin, CourseSettings, TimeStampedModel, HrdbIdModelMixin):
    begin_date_field = 'begin_date'
    end_date_field = 'end_date'
    enroll_begin_field = 'enroll_begin'
    enroll_end_field = 'enroll_end'

    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='groups',
        on_delete=models.PROTECT,
    )
    slug = models.SlugField(_("код"), max_length=255, blank=True)
    name = models.CharField(_("название"), max_length=255)
    summary = models.CharField(_("краткое описание"), max_length=500, blank=True)

    is_active = models.BooleanField(_("активен"), default=True)
    can_join = models.BooleanField(_("можно ли записаться"), default=True)

    num_participants = models.PositiveIntegerField(_("кол-во участников"), default=0, editable=False)
    max_participants = models.PositiveIntegerField(_("макс кол-во участников"), default=0)

    members = models.ManyToManyField(
        User,
        verbose_name=_("участники"),
        related_name='group',
        blank=True,
    )

    tutor = models.ForeignKey(
        Tutor,
        verbose_name=_("наставник"),
        null=True,
        blank=True,
        on_delete=models.PROTECT,
    )

    objects = CourseGroupManager.from_queryset(CourseGroupQuerySet)()

    tracker = FieldTracker(fields=['num_participants', 'max_participants', 'enroll_begin'])

    history = HistoricalRecords()

    class Meta:
        ordering = ('begin_date', 'name')
        verbose_name = _("группа студентов")
        verbose_name_plural = _("группы студентов")
        permissions = (
            ('import_coursegroup', _('Can import course groups')),
            ('export_coursegroup', _('Can export course groups')),
        )

    def __str__(self):
        return f'{self.name} [{self.slug}]' if self.slug else self.name

    def delete(self, *args, **kwargs):
        if self.has_students:
            raise ValidationError(_("нельзя удалить группу, в которой есть студенты"), code='invalid')
        if self.has_enrolled_users:
            raise ValidationError(_("нельзя удалить группу, в которой есть заявки"), code='invalid')

        return super(CourseGroup, self).delete(*args, **kwargs)

    @property
    def was_full(self):
        prev_num_participants = self.tracker.previous('num_participants') or 0
        prev_max_participants = self.tracker.previous('max_participants') or 0
        return prev_max_participants != 0 and prev_num_participants == prev_max_participants

    @property
    def become_not_full_again(self) -> bool:
        return (
            (
                self.tracker.has_changed('num_participants') or
                self.tracker.has_changed('max_participants')
            ) and
            self.was_full and
            not self.is_full
        )

    @property
    def become_available_again(self):
        return self.become_not_full_again and self.available_for_enroll

    def get_current_capacity(self):
        return self.num_participants

    def get_maximum_capacity(self):
        return self.max_participants

    def get_group_or_course_field(self, field_name):
        return getattr(self, field_name, None) or getattr(self.course, field_name, None)

    @property
    def is_full(self) -> bool:
        return self.is_enroll_open and (
            self.max_participants != 0 and self.num_participants >= self.max_participants
        )

    @property
    def available(self):
        return self.is_active and self.can_join

    @property
    def available_for_enroll(self) -> bool:
        return self.can_join and super().available_for_enroll

    @property
    def has_students(self):
        return self.students.exists()

    @property
    def has_enrolled_users(self):
        return self.enrolled_users.exists()

    @property
    def is_enroll_open(self) -> bool:
        return self.available and self.check_enroll_open()

    @property
    def enroll_will_begin(self) -> Optional[datetime]:
        if not self.is_active:
            return None

        now = timezone.now().replace(second=0, microsecond=0)
        if self.enroll_begin is not None and self.enroll_begin > now:
            return self.enroll_begin

        return None


class CourseStudent(TimeStampedModel):
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='students',
        on_delete=models.PROTECT,
    )
    user = models.ForeignKey(
        User,
        verbose_name=_("пользователь"),
        related_name='in_courses',
        on_delete=models.CASCADE,
    )
    group = models.ForeignKey(
        CourseGroup,
        verbose_name=_("группа"),
        related_name='students',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    completion_date = models.DateTimeField(
        _("дата завершения курса"),
        null=True,
        blank=True
    )

    class StatusChoices(models.TextChoices):
        ACTIVE = 'active', _("Активен")
        EXPELLED = 'expelled', _("Отчислен")
        COMPLETED = 'completed', _("Завершил обучение")

    status = models.CharField(
        _("статус"), max_length=20,
        choices=StatusChoices.choices,
        default=StatusChoices.ACTIVE,
    )

    is_passed = models.BooleanField(_("набран проходной балл"), default=False)

    passing_date = models.DateTimeField(
        _("дата прохождения курса"),
        help_text=_("момент времени, когда студент набрал проходной балл"),
        null=True,
        blank=True
    )

    tracker = FieldTracker(fields=['status', 'is_passed'])

    objects = CourseStudentQuerySet.as_manager()

    history = HistoricalRecords()

    class Meta:
        ordering = ('-created',)
        verbose_name = _("студент")
        verbose_name_plural = _("студенты")

    def __str__(self):
        return f'User: {self.user_id} Course: {self.course_id} Group: {self.group_id}'

    def expell(self, change_reason: str = None) -> None:
        if change_reason:
            self._change_reason = change_reason

        self.status = self.StatusChoices.EXPELLED
        self.save()

    def complete(self, change_reason: str = None) -> None:
        if change_reason:
            self._change_reason = change_reason

        self.status = self.StatusChoices.COMPLETED
        self.completion_date = timezone.now()
        self.save()

    def pass_course(self, commit: bool = True) -> None:
        if not self.is_passed:
            self.is_passed = True
        if commit:
            self.save()

    @property
    def can_complete_by_student(self) -> bool:
        """
        Возвращает True,
        если пользователь может завершить данный курс самостоятельно
        :return:
        """
        course = self.course
        if (
            course.structure == Course.StructureChoices.NO_MODULES and
            self.status == self.StatusChoices.ACTIVE
        ):
            return True

        return False

    def clean(self):
        if self.status == CourseStudent.StatusChoices.ACTIVE:
            course_students = CourseStudent.objects.filter(
                course_id=self.course_id, user_id=self.user_id,
            )

            # если обновляем студента - исключаем текущую запись
            if not self._state.adding:
                course_students = course_students.exclude(id=self.id)

            # выдаем ошибку, если отключена мультизапись и есть активные записи студента
            if not self.course.multi_enrollments:
                active_course_students = course_students.filter(status=CourseStudent.StatusChoices.ACTIVE)
                if active_course_students.exists():
                    raise ValidationError(
                        _("пользователь уже является студентом курса"),
                        code='invalid',
                    )

                # выдаем ошибку, если отключено повторное прохождение и есть завершенные курсы
                if not self.course.retries_allowed:
                    completed_course_students = course_students.filter(status=CourseStudent.StatusChoices.COMPLETED)
                    if completed_course_students.exists():
                        raise ValidationError(
                            _("повторное прохождение курса запрещено"),
                            code='invalid',
                        )

    def update_passing_date(self):
        if self.is_passed and not self.passing_date:
            self.passing_date = timezone.now()

    def save(self, *args, **kwargs):
        self.full_clean()
        self.update_passing_date()
        super().save(*args, **kwargs)


class CourseFile(TimeStampedModel):
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='files',
        on_delete=models.PROTECT,
    )
    file = S3UploadField(dest="files", blank=True, null=True)
    filename = models.CharField(_("название"), max_length=255)
    folder = models.CharField(_("директория"), max_length=64, default="default")

    class CourseFileStatus(models.TextChoices):
        PENDING = 'pending', _('в обработке')
        SUCCESS = 'success', _('успех')
        ERROR = 'error', _('ошибка')

    status = models.CharField(
        _("статус"),
        max_length=10,
        choices=CourseFileStatus.choices,
        default=CourseFileStatus.PENDING,
        help_text=_("статус загрузки файла и готовности к использованию"),
    )

    size = models.PositiveIntegerField(_("размер файла"), default=0)
    mimetype = models.CharField(_("тип файла"), max_length=255, blank=True)

    uploaded_by = models.ForeignKey(
        User,
        verbose_name=_("создано"),
        related_name='uploaded_course_files',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    order_with_respect_to = 'course'

    class Meta:
        ordering = ('-created',)
        verbose_name = _("файл для курса")
        verbose_name_plural = _("файлы для курсов")

    def __str__(self):
        return self.filename

    def save(self, *args, **kwargs):
        if not self.file:
            self.file = files_upload_destination(self.filename)
        super().save(*args, **kwargs)


class CourseVisibility(TimeStampedModel):
    course = models.OneToOneField(
        Course,
        verbose_name=_("курс"),
        related_name='visibility',
        on_delete=models.CASCADE,
    )
    rules = JSONField(_("правила"), default=dict)
    formula = models.TextField(_("формула"), blank=True, editable=False)
    parameters = JSONField(_("параметры"), default=dict, editable=False)
    comments = models.TextField(_("комментарии"), blank=True)
    is_active = models.BooleanField(_("активна"), default=True)
    rules_hash = models.CharField(_("хеш"), max_length=64, db_index=True, blank=True, editable=False)

    objects = CourseVisibilityManager()

    class Meta:
        verbose_name = _("правила видимости курса")
        verbose_name_plural = _("правила видимости курсов")

    def __str__(self):
        return str(self.course)

    def parse_rules(self):
        if not getattr(self, '_parsed_rules', None):
            # парсим дерево правил
            try:
                self._parsed_rules = RuleTreeParser(self.rules)
            except RuleParseError as exc:
                logger.exception("rule tree parsing error")
                raise ValidationError({'rules': exc})

        return self._parsed_rules

    def clean(self):
        self.parse_rules()

    def save(self, *args, **kwargs):
        rules = self.parse_rules()
        self.formula = rules.formula
        self.parameters = rules.parameters
        self.rules_hash = UnsaltedSHA1PasswordHasher().encode(str(self.rules), '')
        super().save(*args, **kwargs)

    def available_for(self, user: User) -> bool:
        """
        Проверяет доступность курса для пользователя

        :param user: пользователь
        :return: bool
        """
        return available_for_user(user=user, parameters=self.parameters, formula=self.formula)


class CourseBlock(OrderedModel, TimeStampedModel):
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='blocks',
        on_delete=models.PROTECT,
    )
    name = models.CharField(_("название"), max_length=255)
    summary = models.CharField(_("краткое описание"), max_length=500, blank=True)
    is_active = models.BooleanField(_("активен"), default=True)

    order_with_respect_to = 'course'

    objects = CourseBlockQuerySet.as_manager()

    class Meta:
        ordering = ('order',)
        verbose_name = _("блок курса")
        verbose_name_plural = _("блоки курсов")

    def __str__(self):
        return self.name


class CourseModule(OrderedModel, TimeStampedModel, BaseModule):
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='modules',
        on_delete=models.PROTECT,
    )
    block = models.ForeignKey(
        CourseBlock,
        verbose_name=_("блок"),
        related_name='modules',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    name = models.CharField(_("название"), max_length=255)
    description = models.TextField(_("описание"), blank=True)
    is_active = models.BooleanField(_("активен"), default=True)

    estimated_time = models.PositiveSmallIntegerField(
        _("расчетное время прохождения"),
        null=True,
        blank=True,
        help_text=_("расчетное время прохождения курса (в минутах)")
    )

    weight = models.PositiveSmallIntegerField(
        _("вес"),
        default=0,
        validators=[MaxValueValidator(100)],
    )
    weight_scaled = models.DecimalField(
        _("абсолютная доля веса"),
        max_digits=3,
        decimal_places=2,
        default=0.0,
        validators=[MaxValueValidator(1), MinValueValidator(0)],
        help_text=_("рассчитывается автоматически в зависимости от weight всех модулей в курсе"),
    )

    moe_id = models.CharField(_("id модуля в Мебиусе"), max_length=255, blank=True)

    field_tracker = FieldTracker(fields=['weight', 'is_active'])

    order_with_respect_to = 'course'

    # noinspection PyUnresolvedReferences
    order_class_path = f'{__module__}.CourseModule'

    objects = CourseModuleQuerySet.as_manager()

    class Meta:
        ordering = ('order',)
        verbose_name = _("модуль курса")
        verbose_name_plural = _("модули курсов")

    def set_default_weight(self):
        if not self._state.adding or settings.COURSE_MODULE_DEFAULT_WEIGHT < 0:
            return
        self.weight = min(100, settings.COURSE_MODULE_DEFAULT_WEIGHT)

    def complete(self, student: CourseStudent):
        self.update_progress(student=student, value=100)

    def update_progress(self, student: CourseStudent, value: int, force: bool = False):
        from lms.courses.services import update_module_progress
        update_module_progress(module=self, student=student, value=value, force=force)

    def clean(self):
        if self.block is not None and self.block.course != self.course:
            raise ValidationError(
                _("курс блока должен совпадать с курсом модуля"),
                code='invalid',
            )

    def save(self, *args, **kwargs):
        self.full_clean()
        self.set_default_weight()
        return super().save(*args, **kwargs)

    def __str__(self):
        return self.name


class Cohort(TimeStampedModel):
    class StatusChoices(models.TextChoices):
        PENDING = 'pending', _("в обработке")
        READY = 'ready', _("готова")
        ERROR = 'error', _("ошибка")

    name = models.CharField(_("название"), max_length=255, blank=True)
    status = models.CharField(
        _("статус"),
        max_length=16,
        choices=StatusChoices.choices,
        default=StatusChoices.PENDING,
    )
    error_messages = models.TextField(
        _("ошибки обработки логинов"),
        blank=True,
    )
    logins = JSONField(
        verbose_name=_("список логинов"),
        validators=[list_of_logins_validator],
    )
    users = models.ManyToManyField(
        to=User,
        verbose_name=_("состав когорты"),
        related_name='cohorts',
        blank=True,
    )
    course = models.ForeignKey(
        to=Course,
        verbose_name=_("курс"),
        on_delete=models.CASCADE,
        related_name='cohorts',
        null=True,
        blank=True,
        help_text=_("курс, которому относится когорта (null - глобальная когорта)")
    )
    is_active = models.BooleanField(_("активна"), default=True)

    objects = CohortQuerySet.as_manager()

    tracker = FieldTracker(fields=['logins'])

    history = HistoricalRecords()

    class Meta:
        ordering = ('-course', 'name')
        verbose_name = _("когорта")
        verbose_name_plural = _("когорты")

    def __str__(self):
        if self.course is None:
            return f'{self.name} [{_("глобальная")}]'

        return self.name


class StudentCourseProgress(TimeStampedModel):
    student = models.ForeignKey(
        CourseStudent,
        verbose_name=_("студент"),
        related_name='course_progresses',
        on_delete=models.PROTECT,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='course_progresses',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
        editable=False,
    )
    score = models.PositiveSmallIntegerField(
        _("баллы"),
        default=0,
        validators=[MaxValueValidator(100)]
    )
    history = HistoricalRecords(excluded_fields=['created', 'student', 'course'])

    tracker = FieldTracker(fields=['score'])

    def is_course_passed(self):
        return self.score >= self.course.completion_threshold

    def save(self, *args, **kwargs):
        if not self.course_id:
            self.course_id = self.student.course_id
        if (self._state.adding or self.tracker.has_changed('score')) and self.is_course_passed():
            self.student.pass_course()
        return super().save(*args, **kwargs)

    def __str__(self):
        return _("Прогресс Студента {} по Курсу {}").format(self.student_id, self.course_id)

    class Meta:
        verbose_name = _("прогресс по курсу")
        verbose_name_plural = _("прогресс по курсам")
        constraints = [
            models.UniqueConstraint(fields=['student', 'course'], name='unique_student_course')
        ]


class StudentModuleProgress(TimeStampedModel):
    student = models.ForeignKey(
        CourseStudent,
        verbose_name=_("студент"),
        related_name='module_progresses',
        on_delete=models.PROTECT,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='module_progresses',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
        editable=False,
    )
    module = models.ForeignKey(
        CourseModule,
        verbose_name=_("модуль"),
        related_name='progresses',
        on_delete=models.PROTECT,
    )
    score = models.PositiveSmallIntegerField(
        _("баллы"),
        default=0,
        validators=[MaxValueValidator(100)],
    )
    score_scaled = models.DecimalField(
        _("взвешенные баллы"),
        max_digits=5,
        decimal_places=2,
        default=0.0,
        validators=[MaxValueValidator(100), MinValueValidator(0)],
    )

    tracker = FieldTracker(fields=['score'])

    def clean(self):
        if self.student.course_id != self.module.course_id:
            raise ValidationError(
                _("курс студента должен совпадать с курсом модуля"),
                code='invalid',
            )

    def save(self, *args, **kwargs):
        if not self.course_id:
            self.course = self.student.course
        self.full_clean()
        return super().save(*args, **kwargs)

    def __str__(self):
        return _("Прогресс Студента {} по Модулю {}").format(self.student_id, self.module_id)

    class Meta:
        verbose_name = _("прогресс по модулю")
        verbose_name_plural = _("прогресс по модулям")
        constraints = [
            models.UniqueConstraint(fields=['student', 'module'], name='unique_student_module')
        ]


class LinkedCourse(CourseModule):
    linked_course = models.ForeignKey(
        Course,
        verbose_name=_("вложенный курс"),
        related_name='linked_course_modules',
        on_delete=models.PROTECT,
    )

    class Meta:
        verbose_name = _("Вложенный курс")
        verbose_name_plural = _("Вложенные курсы")

    def clean(self):
        super().clean()
        if self.course.course_type != Course.TypeChoices.TRACK:
            raise ValidationError(
                _("модуль с вложенным курсом можно добавить только в программу"),
                code='invalid',
            )

        if self.linked_course.course_type != Course.TypeChoices.COURSE:
            raise ValidationError(
                _('вложенный курс должен иметь тип "{}"').format(Course.TypeChoices.COURSE.label),
                code='invalid',
            )


class ServiceAccountCourse(models.Model):
    service_account = models.ForeignKey(
        ServiceAccount,
        verbose_name=_("сервисный аккаунт"),
        related_name='courses',
        on_delete=models.PROTECT,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='service_accounts',
        on_delete=models.PROTECT,
    )

    class Meta:
        verbose_name = _("Курс сервисного аккаунта")
        verbose_name_plural = _("Курсы сервисных аккаунтов")
        constraints = [
            models.UniqueConstraint(fields=['service_account', 'course'], name='unique_service_account_per_course')
        ]

    def __str__(self):
        return f'course: {self.course_id} service_account: {self.service_account_id}'
