from guardian.models import GroupObjectPermissionBase, UserObjectPermissionBase

from django.db import models

from rest_framework.permissions import BasePermission, IsAuthenticated

from lms.core.views.permissions import AllowedTVMServiceBasePermission, get_action_permission

from .models import Course, CourseStudent, ServiceAccountCourse


class CourseObjectPermission(BasePermission):
    def has_object_permission(self, request, view, obj):
        course = view.get_course(obj)
        is_course_model = view.get_queryset().model == Course
        permission = get_action_permission(action=view.action, model=Course, is_related=not is_course_model)
        user = view.request.user
        return course.author == user or user.has_perm(permission, course)

    def has_permission(self, request, view):
        user = request.user
        is_course_model = view.get_queryset().model == Course
        permission = get_action_permission(action=view.action, model=Course, is_related=not is_course_model)
        if is_course_model:
            if view.action == 'create':
                return user.has_perm(permission)
        else:
            if view.action in ['create', 'list']:
                course = view.get_course()
                return course.author == user or user.has_perm(permission, course)

        return True


class IsCourseStudent(IsAuthenticated):
    def has_permission(self, request, view):
        if not super().has_permission(request, view):
            return False

        course = view.get_course()

        if not course:
            return False

        return CourseStudent.objects.filter(
            course_id=course.id,
            user_id=request.user.id,
            status__in=[CourseStudent.StatusChoices.ACTIVE, CourseStudent.StatusChoices.COMPLETED],
        ).exists()


class CourseGroupObjectPermission(GroupObjectPermissionBase):
    content_object = models.ForeignKey(Course, on_delete=models.CASCADE)


class CourseUserObjectPermission(UserObjectPermissionBase):
    content_object = models.ForeignKey(Course, on_delete=models.CASCADE)


class CourseAllowedTVMServicePermission(AllowedTVMServiceBasePermission):
    def has_permission(self, request, view):
        tvm_service_id = getattr(request, 'tvm_service_id', None)
        if not tvm_service_id:
            return False

        course_id = view.get_course_id()
        if not course_id:
            return False

        return (ServiceAccountCourse.objects
                .filter(course_id=course_id,
                        service_account__tvm_id=tvm_service_id,
                        service_account__is_active=True)
                .exists())
