from typing import Set

from django.db import models, transaction
from django.dispatch import receiver

from lms.actions.tasks import start_course_passed_trigger_actions_task
from lms.utils.ltree import update_node_path

from .models import (
    Cohort, Course, CourseCategory, CourseGroup, CourseModule, CourseOccupancy, CourseStudent, CourseTeam,
    CourseVisibility, LinkedCourse, StudentModuleProgress,
)
from .services import (
    flush_cache_course_available_for, flush_cache_courses_unavailable_for, refresh_course_module_weight_sum_cache,
    set_course_calc_dates, update_course_on_change_group, update_group_participants, update_occupancy,
    update_permissions_on_add_teams, update_permissions_on_clear_teams, update_permissions_on_remove_teams,
)
from .tasks import (
    process_pending_cohort_task, update_course_progress_task, update_linked_course_module_progress_task,
    update_linked_modules_progress_for_student_task, update_modules_progress_for_student_task,
)


@receiver(signal=models.signals.pre_save, sender=Course)
def course_pre_save_handler(instance: Course, **kwargs):
    set_course_calc_dates(course=instance)


@receiver(signal=models.signals.post_save, sender=Course)
def course_post_save_handler(instance: Course, **kwargs):
    CourseOccupancy.objects.get_or_create(course=instance)


@receiver(signal=models.signals.post_save, sender=CourseCategory)
def course_category_post_save_handler(instance: CourseCategory, **kwargs):
    update_node_path(instance)


@receiver(signal=models.signals.post_save, sender=CourseGroup)
def course_group_post_save_handler(instance: CourseGroup, **kwargs):
    update_occupancy(instance.course)
    update_course_on_change_group(course=instance.course)


@receiver(signal=models.signals.pre_save, sender=CourseGroup)
def course_group_pre_save_handler(instance: CourseGroup, **kwargs):
    update_group_participants(group=instance, commit=False)


@receiver(signal=models.signals.post_delete, sender=CourseGroup)
def course_group_post_delete_handler(instance: CourseGroup, **kwargs):
    update_occupancy(instance.course)
    update_course_on_change_group(course=instance.course)


@receiver(signal=models.signals.m2m_changed, sender=Course.teams.through)
def change_course_permissions_on_change_team_handler(
    sender: Course.teams.through,
    instance: Course,
    action: str,
    reverse: bool,
    model: models.Model,
    pk_set: Set[int],
    **kwargs
):
    if reverse or model is not CourseTeam or sender is not Course.teams.through:
        return

    if action == 'post_add':
        update_permissions_on_add_teams(course=instance, teams_ids=pk_set)
    elif action == 'post_remove':
        update_permissions_on_remove_teams(course=instance, teams_ids=pk_set)
    elif action == 'pre_clear':
        instance._teams_ids = [team.id for team in instance.teams.all()]
    elif action == 'post_clear':
        update_permissions_on_clear_teams(course=instance)


@receiver(signal=models.signals.post_save, sender=CourseVisibility)
def course_visibility_post_save_handler(instance: CourseVisibility, **kwargs):
    flush_cache_courses_unavailable_for()
    flush_cache_course_available_for(courses_ids=[instance.course_id])


@receiver(signal=models.signals.post_delete, sender=CourseVisibility)
def course_visibility_post_delete_handler(instance: CourseVisibility, **kwargs):
    flush_cache_courses_unavailable_for()
    flush_cache_course_available_for(courses_ids=[instance.course_id])


@receiver(signal=models.signals.pre_save, sender=Cohort)
def cohort_pre_save_handler(instance: Cohort, **kwargs):
    if instance.tracker.has_changed('logins'):
        instance.status = Cohort.StatusChoices.PENDING
        instance.error_messages = ''


@receiver(signal=models.signals.post_save, sender=Cohort)
def cohort_post_save_handler(instance: Cohort, **kwargs):
    if instance.status == Cohort.StatusChoices.PENDING:
        transaction.on_commit(lambda: process_pending_cohort_task.delay(cohort_id=instance.id))


@receiver(signal=models.signals.post_save, sender=StudentModuleProgress)
def student_module_progress_post_save_handler(instance: StudentModuleProgress, created: bool, **kwargs):
    if created or instance.tracker.has_changed('score'):
        transaction.on_commit(lambda: update_course_progress_task(student_id=instance.student_id))


@receiver(signal=models.signals.post_save, sender=CourseStudent)
def course_student_post_save_handler(instance: CourseStudent, created, **kwargs):
    if created:
        # Обновление прогресса у модулей курса, на который зачислен студент
        transaction.on_commit(lambda: update_modules_progress_for_student_task.delay(
            student_id=instance.id
        ))
    if instance.is_passed and (created or instance.tracker.has_changed('is_passed')):
        # Обновление прогресса у модулей, связанных с курсом, который завершил пользователь
        transaction.on_commit(lambda: update_linked_modules_progress_for_student_task.delay(
            user_id=instance.user_id,
            course_id=instance.course_id
        ))
        transaction.on_commit(lambda: start_course_passed_trigger_actions_task.delay(
            course_id=instance.course_id,
            student_id=instance.id,
        ))


@receiver(signal=models.signals.post_save, sender=LinkedCourse)
def linked_course_post_save_handler(instance: LinkedCourse, created: bool, **kwargs):
    if created:
        # Обновление прогресса у нового модуля для завершивших вложенный курс студентов
        transaction.on_commit(lambda: update_linked_course_module_progress_task.delay(
            module_id=instance.id
        ))
    if created or instance.field_tracker.has_changed('weight') or instance.field_tracker.has_changed('is_active'):
        transaction.on_commit(lambda: refresh_course_module_weight_sum_cache(course_id=instance.course_id))


@receiver(signal=models.signals.post_delete, sender=LinkedCourse)
def linked_course_post_delete_handler(instance: LinkedCourse, **kwargs):
    transaction.on_commit(lambda: refresh_course_module_weight_sum_cache(course_id=instance.course_id))


# сигнал нужен для случая, когда в джанго-админке меняют параметры модуля в разделе CourseModule
@receiver(signal=models.signals.post_save, sender=CourseModule)
def course_module_post_save_handler(instance: CourseModule, created: bool, **kwargs):
    if instance.field_tracker.has_changed('weight') or instance.field_tracker.has_changed('is_active'):
        transaction.on_commit(lambda: refresh_course_module_weight_sum_cache(course_id=instance.course_id))
