import logging
import traceback

from django.db import transaction
from django.utils.translation import gettext_lazy as _

from config import celery_app as app

from .models import Cohort, CourseStudent, LinkedCourse
from .services import process_pending_cohort, recalculate_all_course_progresses, update_course_progress

log = logging.getLogger(__name__)


@app.task(bind=True)
def process_pending_cohort_task(self, cohort_id: int, force: bool = False, *args, **kwargs) -> None:
    with transaction.atomic():
        cohort = None
        try:
            cohort = Cohort.objects.filter(id=cohort_id).select_for_update(skip_locked=True, of=('self',)).first()
            if cohort is None:
                log.error(_("Cohort %s locked or does not exist") % cohort_id)
                return

            process_pending_cohort(cohort=cohort, force=force)

        except Exception:
            exc_info = traceback.format_exc()
            log.error(exc_info)
            if cohort is not None:
                cohort.status = Cohort.StatusChoices.ERROR
                cohort.error_messages = str(exc_info)
                cohort.save()


@app.task(bind=True)
def process_pending_cohort_periodic_task(self, *args, **kwargs) -> None:
    cohorts = Cohort.objects.filter(status=Cohort.StatusChoices.PENDING)
    for cohort in cohorts:
        process_pending_cohort_task.delay(cohort_id=cohort.id)


@app.task(bind=True)
def update_course_progress_task(self, student_id: int, *args, **kwargs) -> None:
    with transaction.atomic():
        course_student = CourseStudent.objects.filter(id=student_id).select_for_update().first()
        if course_student is None:
            log.error(_("Студент %s не найден") % student_id)
            return
        update_course_progress(student=course_student)


@app.task(bind=True)
def update_linked_course_module_progress_task(self, module_id: int, *args, **kwargs) -> None:
    module = LinkedCourse.objects.get(id=module_id)
    linked_course_completed_users = CourseStudent.objects.filter(
        course_id=module.linked_course_id,
        is_passed=True
    ).values_list('user_id', flat=True)
    course_students = CourseStudent.objects.filter(
        course_id=module.course_id,
        status=CourseStudent.StatusChoices.ACTIVE,
        user_id__in=linked_course_completed_users
    )
    for student in course_students:
        module.complete(student=student)


@app.task(bind=True)
def update_linked_modules_progress_for_student_task(
    self, user_id: int, course_id: int, *args, **kwargs
) -> None:
    """
    Обновляем прогресс по модулям LinkedCourse (завершаем модули), где курс course_id является вложенным.
    Учитываем только такие модули, где юзер является активным студентом
    """
    linked_course_modules = LinkedCourse.objects.filter(
        linked_course_id=course_id,
        course__students__status=CourseStudent.StatusChoices.ACTIVE,
        course__students__user_id=user_id
    )
    students = {}
    for module in linked_course_modules:
        if module.course_id not in students:
            students[module.course_id] = CourseStudent.objects.filter(
                course_id=module.course_id,
                status=CourseStudent.StatusChoices.ACTIVE,
                user_id=user_id
            ).first()
        student = students[module.course_id]
        if student:
            module.complete(student=student)


@app.task(bind=True)
def update_modules_progress_for_student_task(self, student_id: int, *args, **kwargs) -> None:
    student = CourseStudent.objects.get(id=student_id)
    modules = LinkedCourse.objects.filter(
        course_id=student.course_id,
        linked_course__students__user_id=student.user_id,
        linked_course__students__is_passed=True,
    )
    for module in modules:
        module.complete(student=student)


@app.task(bind=True)
def recalculate_all_course_progresses_task(self, course_id, *args, **kwargs) -> None:
    recalculate_all_course_progresses(course_id=course_id)
