from django_filters import rest_framework as filters
from django_filters.constants import EMPTY_VALUES
from drf_spectacular.utils import OpenApiParameter, extend_schema
from guardian.shortcuts import get_perms

from django.core.exceptions import PermissionDenied
from django.db.models import Prefetch
from django.http import Http404
from django.utils.translation import gettext
from django.utils.translation import gettext_lazy as _

from rest_framework import status
from rest_framework.decorators import action
from rest_framework.exceptions import ValidationError
from rest_framework.filters import OrderingFilter
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from lms.core.views.mixins import IsPreviewMixin
from lms.core.views.pagination import LimitOffsetAllPagination
from lms.core.views.viewsets import APIModelViewSet
from lms.users.serializers import PermissionListCodenameSerializer

from ..filtersets import CourseCategoryFilter
from ..models import (
    Course, CourseBlock, CourseCategory, CourseCity, CourseGroup, CourseModule, CourseStudent, LinkedCourse, Provider,
    StudentModuleProgress, StudyMode, Tutor,
)
from ..permissions import IsCourseStudent
from ..serializers import (
    CourseBlockDetailSerializer, CourseBlockListSerializer, CourseCategoryDetailSerializer,
    CourseCategoryListSerializer, CourseCheckSlugSerializer, CourseCityListSerializer, CourseDetailSerializer,
    CourseGroupListSerializer, CourseIDSerializer, CourseListSerializer, CourseModuleDetailSerializer,
    CourseModuleListSerializer, LinkedCourseDetailSerializer, ProviderListSerializer,
    StudentModuleProgressListSerializer, StudyModeListSerializer, TutorDetailSerializer,
)
from ..services import get_unavailable_courses_ids, is_course_available_for_user


class CourseModuleBaseViewSet(APIModelViewSet):
    permission_classes = APIModelViewSet.permission_classes + [IsCourseStudent]
    _cached_course = None

    def get_course(self):
        if not self._cached_course:
            if self.action == 'list':
                self._cached_course = get_object_or_404(Course.objects.active(), pk=self.kwargs.get('pk'))
            elif self.action == 'create':
                self._cached_course = get_object_or_404(Course.objects.active(), pk=self.request.data.get('course_id'))
            else:
                self._cached_course = self.get_object().course
        return self._cached_course


class StudyModeViewSet(APIModelViewSet):
    """
    Список форм обучения
    """
    serializer_class = StudyModeListSerializer
    queryset = StudyMode.objects.active()
    pagination_class = None

    @extend_schema(
        summary=gettext("Список форм обучения")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class ProviderViewSet(APIModelViewSet):
    """
    Список провайдеров - компаний, которые проводят курсы
    """
    serializer_class = ProviderListSerializer
    queryset = Provider.objects.active()
    pagination_class = None

    @extend_schema(
        summary=gettext("Список провайдеров")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class CourseCityViewSet(APIModelViewSet):
    """
    Список городов
    """
    serializer_class = CourseCityListSerializer
    queryset = CourseCity.objects.active()
    pagination_class = None

    # schema_summary = gettext("Cписок городов")

    @extend_schema(
        summary=gettext("Cписок городов")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class CourseCategoryViewSet(APIModelViewSet):
    """
    Список категорий курсов.

    Пока это плоский список, аналогично списку "тем" в бекенде HRDB.
    """
    serializer_classes = {
        'retrieve': CourseCategoryDetailSerializer,
        'list': CourseCategoryListSerializer,
    }
    queryset = CourseCategory.objects.active().select_related('node', 'color_theme').order_by('node__path')
    pagination_class = LimitOffsetAllPagination
    filter_backends = [filters.DjangoFilterBackend]
    filterset_class = CourseCategoryFilter

    @extend_schema(
        summary=gettext("Список категорий курсов"),
        parameters=[
            OpenApiParameter(name='with_courses', required=False, type=bool),
            OpenApiParameter(name='with_available_courses', required=False, type=bool),
        ],
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Информация категории курсов")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class CourseCategoriesViewSet(IsPreviewMixin, APIModelViewSet):
    """
    Список категорий курса.
    """
    serializer_class = CourseCategoryListSerializer
    queryset = CourseCategory.objects.active().select_related('node', 'color_theme')
    pagination_class = LimitOffsetAllPagination

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            user = self.request.user
            qs = Course.objects.active(user=user, preview=self.is_preview)
            get_object_or_404(qs, pk=course_id)
            filter_kwargs = {'courses': self.kwargs['pk']}
            queryset = queryset.filter(**filter_kwargs)
        return queryset

    @extend_schema(
        parameters=[
            OpenApiParameter(name='preview', required=False, type=bool),
        ],
        summary=gettext("Список категорий курса")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class AvailableForEnrollFilter(filters.BooleanFilter):
    def filter(self, qs, value):
        if value in EMPTY_VALUES:
            return qs

        if value is True:
            qs = qs.available_for_enroll()

        return qs


class CourseFilter(filters.FilterSet):
    category = filters.CharFilter(field_name='categories__slug')
    study_mode = filters.CharFilter(field_name='study_mode__slug')
    city = filters.CharFilter(field_name='city__slug')
    name = filters.CharFilter(field_name='name', lookup_expr='icontains')
    payment_method = filters.ChoiceFilter(field_name='payment_method', choices=Course.PaymentMethodChoices.choices)
    course_format = filters.ChoiceFilter(field_name='format', choices=Course.FormatChoices.choices)
    available_for_enroll = AvailableForEnrollFilter()
    type = filters.ChoiceFilter(field_name='course_type', choices=Course.TypeChoices.choices)


class BaseCoursePermissionsViewSet(APIModelViewSet):
    serializer_class = PermissionListCodenameSerializer
    pagination_class = None
    queryset = Course.objects.all()

    @extend_schema(
        summary=gettext("Список разрешений пользователя для курса"),
        responses={200: PermissionListCodenameSerializer},
    )
    def list(self, request, *args, **kwargs):
        user = getattr(request, 'user', None)
        if not user:
            raise Http404

        course = self.get_object()
        perms = get_perms(user, course)
        serializer = self.get_serializer(perms)
        return Response(serializer.data)


class CoursePermissionsViewSet(BaseCoursePermissionsViewSet):
    permission_classes = [
        IsAuthenticated,
    ]


class CourseViewSet(IsPreviewMixin, APIModelViewSet):
    """
    Список курсов
    """
    serializer_class = CourseListSerializer
    queryset = Course.objects.select_related('occupancy', 'author').prefetch_related('categories', 'tags')
    filter_backends = (filters.DjangoFilterBackend,)
    filterset_class = CourseFilter
    permission_classes = [
        IsAuthenticated,
    ]

    serializer_classes = {
        'retrieve': CourseDetailSerializer,
        'list': CourseListSerializer,
    }

    def get_queryset(self):
        qs = super().get_queryset()
        user = getattr(self.request, 'user', None)
        qs = qs.active(user=user, preview=self.is_preview)

        # для detail фильтрация по visibility происходит в get_object
        if self.action == 'list':
            qs = qs.filter(show_in_catalog=True)
            view_permission = qs.get_default_permission('view')
            if user and not user.has_perm(view_permission):
                unavailable_courses_ids = get_unavailable_courses_ids(user=user)
                qs = qs.exclude(pk__in=unavailable_courses_ids)

            opened_groups = Prefetch(
                'groups',
                queryset=CourseGroup.objects.opened(),
                to_attr='_opened_groups',
            )
            qs = qs.prefetch_related(opened_groups)

        else:
            qs = qs.select_related('visibility')

        return qs

    def get_object(self):
        obj = super().get_object()
        user = self.request.user

        if user.is_superuser:
            return obj

        is_visible = True

        # в режиме preview правила видимости не проверяем
        if not self.is_preview:
            is_visible = is_course_available_for_user(course=obj, user=user)

        if not is_visible:
            raise PermissionDenied

        return obj

    @extend_schema(
        summary=gettext("Список курсов"),
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Информация по курсу"),
        parameters=[
            OpenApiParameter(name='preview', required=False, type=bool),
        ],
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class CourseGroupFilter(filters.FilterSet):
    is_full = filters.BooleanFilter(field_name='is_full', method='filter_full')
    available_for_enroll = AvailableForEnrollFilter()
    # available_for_view = AvailableForViewFilter()

    class Meta:
        model = CourseGroup
        fields = [
            'is_full',
            'available_for_enroll',
            # 'available_for_view',
        ]

    def filter_full(self, queryset, name, value):
        """
        Фильтрует группы, в которых есть места, либо в которых
        кол-во мест не ограничено
        """
        if value:
            return queryset

        return queryset.available().not_full()


class CourseGroupViewSet(IsPreviewMixin, APIModelViewSet):
    """
    Список групп по курсу
    """
    serializer_class = CourseGroupListSerializer
    queryset = CourseGroup.objects.active().select_related('tutor', 'tutor__user', 'tutor__user__staffprofile')
    pagination_class = LimitOffsetAllPagination
    permission_classes = [
        IsAuthenticated,
    ]
    filter_backends = (filters.DjangoFilterBackend, OrderingFilter)
    filterset_class = CourseGroupFilter
    ordering_fields = ['begin_date', 'end_date', 'enroll_begin', 'enroll_end', 'name']

    def get_queryset(self):
        queryset = super().get_queryset().available_for_view()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            user = self.request.user
            qs = Course.objects.active(user=user, preview=self.is_preview)
            get_object_or_404(qs, pk=course_id)
            filter_kwargs = {'course_id': self.kwargs['pk']}
            queryset = queryset.filter(**filter_kwargs)
        return queryset

    @extend_schema(
        parameters=[
            OpenApiParameter(name='is_full', required=False, type=bool),
            OpenApiParameter(name='preview', required=False, type=bool),
            OpenApiParameter(name='available_for_enroll', required=False, type=bool),
            # OpenApiParameter(name='available_for_view', required=False, type=bool),
        ],
        summary=gettext("Список групп по курсу"),
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class BaseCourseSlugViewSet(APIModelViewSet):
    """
    Получение ID курса
    """
    serializer_class = CourseIDSerializer
    queryset = Course.objects.only('id')
    lookup_field = 'slug'
    pagination_class = None

    @extend_schema(
        summary=gettext("Получение ID курса")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Валидация кода курса"),
        responses={204: None, 400: None},
    )
    @action(detail=False)
    def validate(self, request, slug):
        serializer = CourseCheckSlugSerializer(data={'slug': slug})
        serializer.is_valid(raise_exception=True)
        return Response(status=status.HTTP_204_NO_CONTENT)


class CourseSlugViewSet(BaseCourseSlugViewSet):
    permission_classes = [
        IsAuthenticated,
    ]


class CourseModuleViewSet(APIModelViewSet):
    serializer_class = CourseModuleListSerializer
    serializer_classes = {
        'retrieve': CourseModuleDetailSerializer,
    }
    queryset = CourseModule.objects.active().select_related('module_type')
    # TODO: доступ только для студентов или предсмотра из лабы
    permission_classes = [
        IsAuthenticated,
    ]
    pagination_class = LimitOffsetAllPagination

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            filter_kwargs = {'course_id': course_id}
            queryset = queryset.filter(**filter_kwargs)
        return queryset

    @extend_schema(
        summary=gettext("Список модулей курса"),
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Информация по модулю"),
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class CourseBlockViewSet(APIModelViewSet):
    serializer_class = CourseBlockListSerializer
    serializer_classes = {
        'retrieve': CourseBlockDetailSerializer,
    }
    queryset = CourseBlock.objects.active()
    permission_classes = [
        IsAuthenticated,
    ]
    pagination_class = LimitOffsetAllPagination

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            filter_kwargs = {'course_id': course_id}
            queryset = queryset.filter(**filter_kwargs)
        return queryset

    @extend_schema(
        summary=gettext("Список блоков курса"),
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Информация по блоку"),
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class TutorViewSet(APIModelViewSet):
    serializer_class = TutorDetailSerializer
    queryset = Tutor.objects.active().select_related('user', 'user__staffprofile')
    pagination_class = None

    @extend_schema(
        summary=gettext("Информация о наставнике")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class CourseStudentCompleteViewSet(APIModelViewSet):
    """
    Завершение курса по инициативе студента
    """
    queryset = (
        CourseStudent.objects
        .filter(status=CourseStudent.StatusChoices.ACTIVE)
        .select_related('course')
    )
    permission_classes = [
        IsAuthenticated,
    ]
    lookup_url_kwarg = 'pk'
    lookup_field = 'course_id'

    def get_queryset(self):
        queryset = super().get_queryset()
        user = getattr(self.request, 'user', None)
        return queryset.filter(user=user)

    @extend_schema(
        request=None,
        responses={204: None},
        summary=gettext("Завершение курса")
    )
    @action(detail=True, methods=['put'])
    def set_complete(self, request, *args, **kwargs):
        instance: CourseStudent = self.get_object()
        if not instance.can_complete_by_student:
            raise ValidationError(
                _("данный курс нельзя завершить самостоятельно"),
                code='cannot_complete',
            )

        instance.complete(change_reason=_("Студент завершил курс самостоятельно"))
        return Response(status=status.HTTP_204_NO_CONTENT)


class StudentModuleProgressViewSet(APIModelViewSet):
    serializer_class = StudentModuleProgressListSerializer
    serializer_classes = {
        'list': StudentModuleProgressListSerializer,
    }
    queryset = StudentModuleProgress.objects.filter(
        student__status=CourseStudent.StatusChoices.ACTIVE,
    )
    permission_classes = [
        IsAuthenticated,
    ]
    pagination_class = LimitOffsetAllPagination

    def get_course(self, pk: int):
        queryset = Course.objects.active()
        return get_object_or_404(queryset, pk=pk)

    def get_queryset(self):
        queryset = super().get_queryset()
        user = getattr(self.request, 'user')
        queryset = queryset.filter(student__user=user)

        if self.action == 'list':
            course_id = self.kwargs['pk']
            self.get_course(course_id)
            filter_kwargs = {'course_id': course_id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset


class LinkedCourseViewSet(CourseModuleBaseViewSet):
    """
    Вложенный курс
    """
    serializer_class = LinkedCourseDetailSerializer
    queryset = LinkedCourse.objects.active().select_related('linked_course')

    @extend_schema(
        summary=gettext("Информация о вложенном курсе")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class CourseModuleStudentCompleteViewSet(APIModelViewSet):
    """
    Завершение модуля по инициативе студента
    """
    queryset = CourseModule.objects.active().only('module_type')
    permission_classes = [
        IsCourseStudent,
    ]
    _cached_queryset = None
    _cached_module = None

    def get_queryset(self):
        if self._cached_queryset is None:
            queryset = super().get_queryset()
            module = queryset.filter(pk=self.kwargs.get('pk')).first()
            self._cached_queryset = queryset.none() if not module else module.module_type.get_all_objects_for_this_type(
                is_active=True
            ).select_related('course')
        return self._cached_queryset

    def get_object(self):
        if not self._cached_module:
            self._cached_module = super().get_object()
        return self._cached_module

    def get_course(self):
        return self.get_object().course

    @extend_schema(
        request=None,
        responses={204: None},
        summary=gettext("Завершение модуля")
    )
    @action(detail=True, methods=['put'])
    def set_complete(self, request, *args, **kwargs):
        module = self.get_object()
        student = get_object_or_404(
            CourseStudent.objects.active().select_related('course'),
            course_id=module.course_id, user=self.request.user,
        )
        module.complete(student=student)
        return Response(status=status.HTTP_204_NO_CONTENT)
