from django_admin_listfilter_dropdown.filters import RelatedOnlyDropdownFilter
from import_export.admin import ImportExportMixin
from ordered_model.admin import OrderedTabularInline
from simple_history.admin import SimpleHistoryAdmin

from django.contrib import admin
from django.utils.translation import gettext_lazy as _

from lms.core.admin.mixins import AddFieldsetsMixin, AddReadonlyFieldsMixin, AdminChangeLinksMixin

from .import_export_resources import EnrolledUserResource, EnrollmentResource
from .models import EnrolledUser, Enrollment, EnrollSurvey, EnrollSurveyField


@admin.register(Enrollment)
class EnrollmentAdmin(ImportExportMixin, admin.ModelAdmin):
    list_display = (
        'name', 'enroll_type', 'survey', 'course',
        'is_active', 'is_default', 'hrdb_id',
        'created', 'modified',
    )
    list_select_related = ('course',)
    list_filter = (
        'is_active',
        'enroll_type',
        ('course', RelatedOnlyDropdownFilter),
    )
    search_fields = (
        'name',
        'course__id',
        'course__name',
        'course__slug',
    )
    raw_id_fields = ('course',)

    fields = (
        'course', 'enroll_type', 'survey', 'name', 'summary',
        'options', 'hrdb_id', 'is_active', 'is_default',
    )
    resource_class = EnrollmentResource


@admin.register(EnrolledUser)
class EnrolledUserAdmin(
    ImportExportMixin, AdminChangeLinksMixin, AddFieldsetsMixin, AddReadonlyFieldsMixin, SimpleHistoryAdmin,
):
    list_display = (
        'user', 'course_link', 'enrollment', 'group_link', 'status',
        'display_group_capacity', 'hrdb_id', 'created', 'modified',
    )
    list_select_related = ('course', 'course__author', 'group', 'user', 'enrollment', 'course_student')
    list_filter = (
        'status',
        ('course', RelatedOnlyDropdownFilter),
        ('group', RelatedOnlyDropdownFilter),
    )
    search_fields = (
        'user__id',
        'user__username',
        'user__yauid',
        'course__id',
        'course__name',
        'group__id',
        'group__name',
    )
    raw_id_fields = ('user', 'course', 'enrollment', 'group', 'course_student')
    add_fieldsets = (
        (None, {'fields': [
            'course', 'enrollment', 'user', 'group', 'survey', 'status', 'comments', 'course_student',
        ]}),
        (_("Ответы по анкете"), {'fields': [
            'survey_data',
        ]}),
    )
    fieldsets = (
        (None, {'fields': [
            'course_link', 'enrollment_link', 'user_link', 'group_link', 'survey_link', 'course_student_link',
            'status', 'comments', 'groups', 'created', 'modified',
        ]}),
        (_("Ответы по анкете"), {'fields': [
            'survey_data', 'user_answers', 'hrdb_survey_data',
        ]}),
        (_("Даты"), {'fields': [
            'enroll_date', 'completion_date',
        ]}),
    )
    add_readonly_fields = (
        'survey', 'user_answers', 'hrdb_survey_data', 'groups',
        'course_link', 'user_link', 'group_link', 'survey_link', 'course_student_link',
    )
    readonly_fields = add_readonly_fields + (
        'course', 'enrollment', 'user', 'group', 'course_student',
        'created', 'modified',
    )

    change_links = [
        'course', 'enrollment', 'user', 'group', 'survey', 'course_student',
    ]
    resource_class = EnrolledUserResource

    def user_link_label(self, obj):
        fullname = obj.get_full_name()
        return '{} ({})'.format(fullname, obj) if fullname else obj

    def display_group_capacity(self, obj: 'EnrolledUser'):
        if not obj.group:
            return ''
        return '{}/{}'.format(
            obj.group.num_participants,
            obj.group.max_participants if obj.group.max_participants > 0 else '-',
        )

    display_group_capacity.short_description = _("Кол-во в группе")


class EnrollSurveyFieldInline(OrderedTabularInline):
    model = EnrollSurveyField
    fields = ('name', 'field_type', 'title', 'parameters',
              'default_value', 'is_required', 'is_hidden',
              'description', 'placeholder',)
    extra = 0


@admin.register(EnrollSurvey)
class EnrollSurveyAdmin(admin.ModelAdmin):
    list_display = ('slug', 'name', 'is_active', 'created_by', 'created', 'modified')
    list_select_related = ('created_by', )
    list_filter = ('is_active',)
    raw_id_fields = ('created_by',)

    inlines = [
        EnrollSurveyFieldInline,
    ]
