from tqdm import tqdm

from django.conf import settings
from django.core.management.base import BaseCommand
from django.db import transaction

from lms.courses.models import CourseStudent
from lms.enrollments.models import EnrolledUser


class DryRunException(Exception):
    pass


class Command(BaseCommand):
    def get_student(self, enrolled_user: EnrolledUser):
        return CourseStudent.objects.filter(
            course=enrolled_user.course,
            group=enrolled_user.group,
            user=enrolled_user.user,
        ).first()

    def fill_course_student(self, options):
        enrolled_users = EnrolledUser.objects.filter(course_student__isnull=True)
        enrolled_users_to_update = []
        enrolled_users_to_update_ids = []
        try:
            with transaction.atomic():
                for enrolled_user in tqdm(enrolled_users, total=enrolled_users.count()):
                    course_student = self.get_student(enrolled_user=enrolled_user)
                    if course_student is not None:
                        enrolled_user.course_student = course_student
                        enrolled_users_to_update.append(enrolled_user)
                        enrolled_users_to_update_ids.append(enrolled_user.id)

                EnrolledUser.objects.bulk_update(
                    objs=enrolled_users_to_update,
                    fields=['course_student'],
                    batch_size=settings.BULK_BATCH_SIZE_DEFAULT,
                )

                if options.get('dry_run', False):
                    raise DryRunException

        except DryRunException:
            self.stdout.write('Rollback')
        else:
            self.stdout.write('Commit')

        self.stdout.write(
            f"""
                Enrolled users to update: {len(enrolled_users_to_update_ids)}
                {enrolled_users_to_update_ids}
            """
        )

    def add_arguments(self, parser):
        parser.add_argument(
            '--dry-run', action='store_true', dest='dry_run',
            help='Run without commit to db',
        )

    def handle(self, *args, **options):

        self.fill_course_student(options)

        self.stdout.write("Done\n")
