from functools import partial

from django.contrib.auth import get_user_model

from rest_framework import serializers

from lms.core.validators import validate_unique_elements

from ..models import EnrollSurvey, EnrollSurveyField
from ..survey.datasets import DATASETS
from ..survey.validators import dataset_validator

User = get_user_model()


# API Serializers
# ===============
class EnrollSurveyFieldDetailSerializer(serializers.ModelSerializer):
    parameters = serializers.JSONField(source='expanded_parameters')

    class Meta:
        model = EnrollSurveyField
        fields = (
            'id', 'survey', 'name', 'field_type', 'title', 'description',
            'placeholder', 'parameters', 'default_value', 'is_required', 'is_hidden',
        )
        read_only_fields = fields


class EnrollSurveyDetailSerializer(serializers.ModelSerializer):
    fields = EnrollSurveyFieldDetailSerializer(many=True)

    class Meta:
        model = EnrollSurvey
        fields = (
            'id', 'slug', 'name', 'summary', 'description', 'fields',
        )
        read_only_fields = fields


class ChoiceListField(serializers.ListField):
    child = serializers.ChoiceField(choices=[])


class EnrollSurveyDataSerializer(serializers.Serializer):
    field_mappings = {
        EnrollSurveyField.TYPE_TEXT: serializers.CharField,
        EnrollSurveyField.TYPE_TEXTAREA: serializers.CharField,
        EnrollSurveyField.TYPE_NUMBER: serializers.IntegerField,
        EnrollSurveyField.TYPE_URL: serializers.URLField,
        EnrollSurveyField.TYPE_EMAIL: serializers.EmailField,
        EnrollSurveyField.TYPE_SELECT: serializers.ChoiceField,
        EnrollSurveyField.TYPE_DATASET: serializers.CharField,
        EnrollSurveyField.TYPE_MULTICHECKBOX: ChoiceListField,
    }
    parameters_mapping = {
        EnrollSurveyField.TYPE_TEXT: ['max_length'],
        EnrollSurveyField.TYPE_TEXTAREA: ['max_length'],
        EnrollSurveyField.TYPE_NUMBER: ['max_value', 'min_value'],
        EnrollSurveyField.TYPE_URL: ['max_length'],
        EnrollSurveyField.TYPE_EMAIL: ['max_length'],
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        survey = self.context.get('survey', None)  # type: EnrollSurvey

        for survey_field in survey.fields.all():  # type: EnrollSurveyField
            field = self.field_mappings.get(survey_field.field_type, serializers.CharField)

            field_kwargs = {
                'label': survey_field.title,
                'required': survey_field.is_required,
            }

            if survey_field.default_value:
                field_kwargs['default'] = survey_field.default_value
                field_kwargs['required'] = False

            if issubclass(field, (serializers.CharField, serializers.ChoiceField)):
                field_kwargs['allow_blank'] = not survey_field.is_required

            if survey_field.field_type in self.parameters_mapping:
                field_kwargs.update(
                    {
                        parameter_name: survey_field.parameters.get(parameter_name)
                        for parameter_name
                        in self.parameters_mapping[survey_field.field_type]
                        if survey_field.parameters.get(parameter_name)
                    }
                )
            elif survey_field.field_type == EnrollSurveyField.TYPE_SELECT:
                field_kwargs['choices'] = survey_field.options.keys()
            elif survey_field.field_type == EnrollSurveyField.TYPE_DATASET:
                dataset = DATASETS[survey_field.parameters['dataset']['name']]
                validator = partial(dataset_validator, dataset=dataset)
                field_kwargs['validators'] = [validator]

            elif survey_field.field_type == EnrollSurveyField.TYPE_MULTICHECKBOX:
                field_kwargs['child'] = serializers.ChoiceField(choices=survey_field.options.keys())
                field_kwargs['validators'] = [validate_unique_elements]
                field_kwargs['allow_null'] = not survey_field.is_required
                field_kwargs['allow_empty'] = True  # "Любой"

            self.fields[survey_field.name] = field(**field_kwargs)


# LabAPI Serializers
# ==================
class EnrollSurveyCreatorLabSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = (
            'id', 'username', 'first_name', 'last_name',
        )
        read_only_fields = fields


class EnrollSurveyFieldDetailLabSerializer(serializers.ModelSerializer):
    class Meta:
        model = EnrollSurveyField
        fields = (
            'id', 'survey_id', 'name', 'field_type', 'title', 'description',
            'placeholder', 'parameters', 'default_value', 'is_required', 'is_hidden',
        )
        read_only_fields = fields


class EnrollSurveyDetailLabSerializer(serializers.ModelSerializer):
    created_by = EnrollSurveyCreatorLabSerializer(read_only=True)
    fields = EnrollSurveyFieldDetailLabSerializer(many=True)

    class Meta:
        model = EnrollSurvey
        fields = (
            'id', 'slug', 'name', 'summary', 'description', 'is_active', 'created_by', 'fields',
        )
        read_only_fields = fields


class EnrollSurveyListLabSerializer(serializers.ModelSerializer):
    created_by = EnrollSurveyCreatorLabSerializer(read_only=True)

    class Meta:
        model = EnrollSurvey
        fields = (
            'id', 'slug', 'name', 'summary', 'description', 'is_active', 'created_by',
        )
