import enum
from typing import Optional

from django.contrib.auth import get_user_model
from django.db.models import F
from django.utils.translation import gettext_lazy as _

from lms.courses.models import CourseGroup, CourseOccupancy, CourseStudent

from .models import EnrolledUser, Enrollment

User = get_user_model()


class ChangeOperation(enum.IntEnum):
    INCREASE = 1
    DECREASE = -1


def _create_course_student(enrolled_user: EnrolledUser):
    """
    Создает студента или изменяет уже существующего
    после перехода заявки в статус `ENROLLED`
    """
    if enrolled_user.course_student_id is None:
        course_student = CourseStudent(
            user_id=enrolled_user.user_id,
            course_id=enrolled_user.course_id,
            group_id=enrolled_user.group_id,
            status=CourseStudent.StatusChoices.ACTIVE,
        )
        course_student._change_reason = _("cоздано при подтверждении заявки")
    else:
        course_student = enrolled_user.course_student
        course_student.status = CourseStudent.StatusChoices.ACTIVE
        course_student._change_reason = _("изменено при подтверждении заявки")

    course_student.save()
    return course_student


def update_or_create_course_student(enrolled_user: EnrolledUser) -> None:
    """
    Создание/обновление студента на курсе

    :param enrolled_user:
    :return:
    """
    if not enrolled_user.tracker.has_changed('status'):
        return

    # при подтверждении заявки создаем студента, если ещё не создан
    if enrolled_user.status == EnrolledUser.StatusChoices.ENROLLED:
        course_student = _create_course_student(enrolled_user)
        EnrolledUser.objects.filter(id=enrolled_user.id).update(course_student=course_student)

    # при отклонении заявки - ставим у студента статус "отчислен", если студент существует
    elif enrolled_user.status == EnrolledUser.StatusChoices.REJECTED:
        course_student: Optional[CourseStudent] = enrolled_user.course_student
        if course_student and course_student.status != CourseStudent.StatusChoices.EXPELLED:
            course_student.expell(change_reason=_("отчисление при отклонении заявки"))

    # при завершении заявки - ставим у студента статус "завершил обучение", если студент существует
    elif enrolled_user.status == EnrolledUser.StatusChoices.COMPLETED:
        course_student: Optional[CourseStudent] = enrolled_user.course_student
        if course_student and course_student.status != CourseStudent.StatusChoices.COMPLETED:
            course_student.complete(change_reason=_("завершение обучения при завершении заявки"))


def delete_course_and_group_participants(enrolled_user: EnrolledUser) -> None:
    """
    Обновление участников курса и групп, при УДАЛЕНИИ заявки

    :param enrolled_user:
    :return:
    """
    if enrolled_user.status in EnrolledUser.USER_STATUS_KEYS:
        _update_occupancy_and_group_participants(enrolled_user, ChangeOperation.DECREASE)


def update_course_and_group_participants(enrolled_user: EnrolledUser) -> None:
    """
    Обновление участников курса и групп, при ОБНОВЛЕНИИ заявки

    :param enrolled_user:
    :return:
    """
    op = None

    if not enrolled_user.tracker.has_changed('status'):
        return

    previous_status = enrolled_user.tracker.previous('status')
    need_statuses = EnrolledUser.USER_STATUS_KEYS

    # если заявка была поставлена в статус PENDING/ENROLLED -
    # увеличиваем число участников
    if previous_status not in need_statuses and enrolled_user.status in need_statuses:
        op = ChangeOperation.INCREASE

    # если заявки была поставлена из статуса PENDING/ENROLLED в другой статус -
    # уменьшаем число участников
    if previous_status in need_statuses and enrolled_user.status not in need_statuses:
        op = ChangeOperation.DECREASE

    if op:
        _update_occupancy_and_group_participants(enrolled_user, op)


def _update_occupancy_and_group_participants(enrolled_user: EnrolledUser, op: ChangeOperation) -> None:
    """
    Обновление заполненности курса и кол-во участников групп

    :param enrolled_user:
    :param op:
    :return:
    """
    group_id = enrolled_user.group_id
    if group_id:
        filter_kwargs = {'id': group_id}
        if op == ChangeOperation.DECREASE:
            filter_kwargs['num_participants__gte'] = 1
        CourseGroup.objects.filter(**filter_kwargs).update(num_participants=F("num_participants") + op.value)

    course_id = enrolled_user.course_id
    CourseOccupancy.objects.update_or_create(
        course=course_id,
        defaults={'current': F("current") + op.value},
    )


def update_enrolled_user_groups(enrolled_user: EnrolledUser) -> None:
    """
    Обновление строки с подразделениями пользователя

    :param enrolled_user:
    :return:
    """
    if not enrolled_user.groups:
        staffprofile = getattr(enrolled_user.user, 'staffprofile', None)
        if staffprofile:
            enrolled_user.groups = staffprofile.groups_str()


def enroll_type_instant_change_status(enrolled_user: EnrolledUser) -> None:
    """
    Обновление статуса заявки при автоматическом зачислении

    :param enrolled_user:
    :return:
    """
    if enrolled_user._state.adding and enrolled_user.enrollment.enroll_type == Enrollment.TYPE_INSTANT:
        enrolled_user.status = EnrolledUser.StatusChoices.ENROLLED


def update_default_enrollment(enrollment: Enrollment) -> None:
    """
    Обновление дефолтного механизма зачисления на курсе

    Если текущий механизм зачисления установлен как дефолтный,
    проверяем и сбрасывает признак is_default у остальных механизмов в курсе

    :param enrollment:
    :return:
    """
    if enrollment.is_default:
        Enrollment.objects.filter(
            course_id=enrollment.course_id,
        ).exclude(
            id=enrollment.id,
        ).update(is_default=False)
