from typing import Any

from django.core.exceptions import ValidationError
from django.db.models.query import QuerySet
from django.utils.translation import gettext_lazy as _

from rest_framework.generics import ListAPIView

from lms.core.views.pagination import LimitOffsetAllPagination

from .serializers import DatasetSerializer


class AbstractDataset:
    def get_queryset(self) -> QuerySet:
        raise NotImplementedError

    def as_view(self, *args, **kwargs):
        raise NotImplementedError

    def validate(self, value: Any) -> Any:
        raise NotImplementedError


class DatasetView(ListAPIView):
    serializer_class = DatasetSerializer
    pagination_class = LimitOffsetAllPagination

    dataset = None

    def get_queryset(self) -> QuerySet:
        if not hasattr(self, 'dataset'):
            raise ValueError(_('не указан датасет'))

        return self.dataset.get_queryset()


class BaseDataset(AbstractDataset):
    view_class = DatasetView
    lookup_field = 'pk'

    def validate(self, value: Any) -> Any:
        filter_kwargs = {self.lookup_field: value}
        if self.get_queryset().filter(**filter_kwargs).exists():
            return value

        raise ValidationError(
            _('{model_name} с {lookup_field}={value} не найден').format(
                model_name=self.get_queryset().model._meta.verbose_name,
                lookup_field=self.lookup_field,
                value=value,
            ),
            code='invalid',
        )

    def as_view(self, *args, **kwargs):
        kwargs['dataset'] = self
        view = self.view_class()
        return view.as_view(*args, **kwargs)


class CourseCategoriesDataset(BaseDataset):
    def get_queryset(self) -> QuerySet:
        from lms.courses.models import CourseCategory
        return CourseCategory.objects.active()


class TrackerCourseCategoryDataset(BaseDataset):
    def get_queryset(self) -> QuerySet:
        from lms.tracker.models import TrackerCourseCategory
        return TrackerCourseCategory.objects.all()


class CourseDataset(BaseDataset):
    def get_queryset(self) -> QuerySet:
        from lms.courses.models import Course
        return Course.objects.active()


DATASETS = {
    'course_categories': CourseCategoriesDataset(),
    'tracker_course_categories': TrackerCourseCategoryDataset(),
    'courses': CourseDataset(),
}
