from typing import Any

import jsonschema

from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _

from .datasets import DATASETS, BaseDataset


class BaseSchemaValidator:
    schema = {
        "type": "object",
    }
    code = 'invalid'

    def get_schema(self):
        return self.schema

    def __call__(self, value):
        try:
            jsonschema.validate(value, self.get_schema())
        except jsonschema.exceptions.ValidationError as exc:
            raise ValidationError(message=str(exc), code=self.code)


class FieldSchemaValidator(BaseSchemaValidator):
    schema = {
        "type": "object",
        "properties": {
            "max_length": {
                "type": "integer",
                "minimum": 1,
            },
        },
    }


class NumberFieldSchemaValidator(BaseSchemaValidator):
    schema = {
        "type": "object",
        "properties": {
            "min_value": {"type": "integer"},
            "max_value": {"type": "integer"},
        },
    }

    def __call__(self, value):
        errors = []
        try:
            super().__call__(value)
        except ValidationError as exc:
            errors.append(exc)

        min_value = value.get('min_value', None)
        max_value = value.get('max_value', None)

        if isinstance(min_value, int) and isinstance(max_value, int) and min_value > max_value:
            errors.append(
                ValidationError(
                    _('min_value must be less than max_value'),
                    code='invalid',
                )
            )

        if errors:
            raise ValidationError(errors)


class SelectFieldSchemaValidator(BaseSchemaValidator):
    schema = {
        "type": "object",
        "properties": {
            "options": {
                "type": "array",
                "minItems": 2,
                "items": {
                    "type": "object",
                    "properties": {
                        "value": {"type": "string"},
                        "content": {"type": "string"},
                    },
                    "required": ["value", "content"],
                },
            },
        },
        "required": ["options"],
    }


class DatasetFieldSchemaValidator(BaseSchemaValidator):
    schema = {
        "type": "object",
        "properties": {
            "dataset": {
                "type": "object",
                "properties": {
                    "name": {
                        "type": "string",
                        "enum": list(DATASETS.keys()),
                    },
                },
                "required": ["name"],
            }
        },
        "required": ["dataset"],
    }


class MulticheckboxFieldSchemaValidator(BaseSchemaValidator):
    schema = {
        "type": "object",
        "properties": {
            "options": {
                "type": "array",
                "minItems": 2,
                "items": {
                    "type": "array",
                    "minItems": 1,
                    "items": {
                        "type": "object",
                        "properties": {
                            "value": {"type": "string"},
                            "content": {"type": "string"},
                        },
                        "required": ["value", "content"],
                    },
                },
            },
        },
        "required": ["options"],
    }


def dataset_validator(value: Any, dataset: BaseDataset) -> Any:
    dataset.validate(value)

    return value
