from django_filters import rest_framework as filters
from django_filters.constants import EMPTY_VALUES
from django_filters.rest_framework import DjangoFilterBackend
from drf_spectacular.utils import extend_schema

from django.utils.translation import gettext

from rest_framework.generics import get_object_or_404

from lms.core.views.viewsets import LabModelViewSet
from lms.courses.models import Course
from lms.courses.permissions import CourseObjectPermission

from ..models import EnrolledUser, Enrollment, EnrollSurvey
from ..serializers import (
    EnrollmentCreateLabSerializer, EnrollmentCreateWithoutTypeLabSerializer, EnrollmentDetailLabSerializer,
    EnrollmentListLabSerializer, EnrollmentUpdateLabSerializer, EnrollSurveyDetailLabSerializer,
    EnrollSurveyListLabSerializer,
)


class EnrollSurveyLabViewSet(LabModelViewSet):
    serializer_class = EnrollSurveyListLabSerializer
    serializer_classes = {
        'retrieve': EnrollSurveyDetailLabSerializer,
    }
    queryset = EnrollSurvey.objects.active().select_related('created_by')
    pagination_class = None

    @extend_schema(
        summary=gettext("Список анкет при зачислении")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Информация при зачислении")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class EnrollmentLabViewSet(LabModelViewSet):
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]
    serializer_class = EnrollmentListLabSerializer
    serializer_classes = {
        'create': EnrollmentCreateLabSerializer,
        'update': EnrollmentUpdateLabSerializer,
        'partial_update': EnrollmentUpdateLabSerializer,
        'retrieve': EnrollmentDetailLabSerializer,
    }
    pagination_class = None
    queryset = Enrollment.objects.all()
    filter_backends = [DjangoFilterBackend]
    filterset_fields = ['is_active', 'is_default']

    def get_course(self, obj=None):
        if obj is not None:
            return obj.course
        if self.action == 'create':
            return get_object_or_404(Course.objects.all(), pk=self.request.data.get('course_id'))
        if self.action == 'list':
            return get_object_or_404(Course.objects.all(), pk=self.kwargs.get('pk'))

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            filter_kwargs = {'course_id': course_id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset

    @extend_schema(
        summary=gettext("Список механизмов зачисления")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        responses={200: EnrollmentDetailLabSerializer},
        summary=gettext("Информация о механизме зачисления")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentCreateLabSerializer,
        responses={201: EnrollmentDetailLabSerializer},
        summary=gettext("Создание механизма зачисления"),
    )
    def create(self, request, *args, **kwargs):
        return super().create(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentUpdateLabSerializer,
        responses={200: EnrollmentDetailLabSerializer},
        summary=gettext("Обновление механизма зачисления"),
    )
    def update(self, request, *args, **kwargs):
        return super().update(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentUpdateLabSerializer,
        responses={200: EnrollmentDetailLabSerializer},
        summary=gettext("Частичное обновление механизма зачисления"),
    )
    def partial_update(self, request, *args, **kwargs):
        return super().partial_update(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление механизма зачисления"),
    )
    def destroy(self, request, *args, **kwargs):
        return super().destroy(request, *args, **kwargs)


class EnrolledUserLabFilterSet(filters.FilterSet):
    department_id = filters.NumberFilter(method='filter_department_id')
    logins = filters.CharFilter(method='filter_logins', label=gettext("список логинов, раздеденных запятыми"))
    group_name = filters.CharFilter(field_name='group', lookup_expr='name')

    class Meta:
        model = EnrolledUser
        fields = [
            'group_id', 'group_name', 'status', 'user__username',
            'department_id', 'logins', 'enrollment_tracker_issues__status',
            'enroll_date', 'completion_date', 'created', 'modified',
        ]

    def filter_department_id(self, queryset, _, value):
        if value in EMPTY_VALUES:
            return queryset

        if value:
            return queryset.filter(user__staffprofile__groups_tree__contains=value)

        return queryset

    def filter_logins(self, queryset, _, value):
        if value in EMPTY_VALUES:
            return queryset

        return queryset.filter(user__username__in=map(str.strip, value.split(',')))


class EnrollmentLabViewSetBase(EnrollmentLabViewSet):
    serializer_class = EnrollmentDetailLabSerializer
    serializer_classes = {
        'retrieve': EnrollmentDetailLabSerializer,
        'create': EnrollmentCreateWithoutTypeLabSerializer,
        'update': EnrollmentUpdateLabSerializer,
        'partial_update': EnrollmentUpdateLabSerializer,
    }

    def get_queryset(self, *args, **kwargs):
        return super().get_queryset(*args, **kwargs).filter(enroll_type=self.ENROLL_TYPE)

    def perform_create(self, serializer):
        serializer.save(enroll_type=self.ENROLL_TYPE)


class EnrollmentInstantLabViewSet(EnrollmentLabViewSetBase):
    ENROLL_TYPE = Enrollment.TYPE_INSTANT

    @extend_schema(
        summary=gettext("Информация по автоматическому методу зачисления на курс")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentCreateWithoutTypeLabSerializer,
        responses={201: EnrollmentDetailLabSerializer},
        summary=gettext("Создание автоматического метода зачисления"),
    )
    def create(self, request, *args, **kwargs):
        return super().create(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentUpdateLabSerializer,
        responses={200: EnrollmentDetailLabSerializer},
        summary=gettext("Обновление автоматического метода зачисления"),
    )
    def update(self, request, *args, **kwargs):
        return super().update(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentUpdateLabSerializer,
        responses={200: EnrollmentDetailLabSerializer},
        summary=gettext("Частичное обновление автоматического метода зачисления"),
    )
    def partial_update(self, request, *args, **kwargs):
        return super().partial_update(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление автоматического метода зачисления"),
        responses={204: None},
    )
    def destroy(self, request, *args, **kwargs):
        return super().destroy(request, *args, **kwargs)


class EnrollmentManualLabViewSet(EnrollmentLabViewSetBase):
    ENROLL_TYPE = Enrollment.TYPE_MANUAL

    @extend_schema(
        summary=gettext("Информация по ручному методу зачисления на курс")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentCreateWithoutTypeLabSerializer,
        responses={201: EnrollmentDetailLabSerializer},
        summary=gettext("Создание ручного метода зачисления"),
    )
    def create(self, request, *args, **kwargs):
        return super().create(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentUpdateLabSerializer,
        responses={200: EnrollmentDetailLabSerializer},
        summary=gettext("Обновление ручного метода зачисления"),
    )
    def update(self, request, *args, **kwargs):
        return super().update(request, *args, **kwargs)

    @extend_schema(
        request=EnrollmentUpdateLabSerializer,
        responses={200: EnrollmentDetailLabSerializer},
        summary=gettext("Частичное обновление ручного метода зачисления"),
    )
    def partial_update(self, request, *args, **kwargs):
        return super().partial_update(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление ручного метода зачисления"),
        responses={204: None},
    )
    def destroy(self, request, *args, **kwargs):
        return super().destroy(request, *args, **kwargs)
