from model_utils.models import TimeStampedModel

from django.core.exceptions import ValidationError
from django.db import models
from django.utils.translation import gettext_lazy as _

from lms.courses.models import CourseTeam
from lms.users.models import Group

from .constants import RESERVED_RULE_NAMES


class FirewallRule(TimeStampedModel):
    slug = models.SlugField(_("slug"), max_length=255, unique=True)
    summary = models.CharField(_("аннотация"), max_length=255, blank=True)

    class Meta:
        ordering = ('slug',)
        verbose_name = _("Правило firewall")
        verbose_name_plural = _("Правила firewall")

    def clean(self):
        if self.slug in RESERVED_RULE_NAMES:
            raise ValidationError({
                'slug': _("Данное имя зарезервировано. Пожалуйста, выберите другое"),
            })

    def __str__(self):
        return self.slug


class GroupFirewallRule(TimeStampedModel):
    group = models.OneToOneField(
        Group,
        verbose_name=_("группа"),
        related_name='group_firewall_rule',
        primary_key=True,
        on_delete=models.CASCADE,
    )
    rule = models.ForeignKey(
        FirewallRule,
        verbose_name=_("правило"),
        related_name='groups',
        on_delete=models.PROTECT,
    )
    is_staff = models.BooleanField(_("Доступ в django-админку"), default=False)

    class Meta:
        ordering = ('group',)
        verbose_name = _("Правило для групп")
        verbose_name_plural = _("Правила для групп")

    def __str__(self):
        return f'{self.group_id} {self.rule}'


class TeamFirewallRule(TimeStampedModel):
    team = models.OneToOneField(
        CourseTeam,
        verbose_name=_("команда"),
        related_name='team_firewall_rule',
        primary_key=True,
        on_delete=models.CASCADE,
    )
    rule = models.ForeignKey(
        FirewallRule,
        verbose_name=_("правило"),
        related_name='teams',
        on_delete=models.PROTECT,
    )

    class Meta:
        ordering = ('team',)
        verbose_name = _("Правило для команд")
        verbose_name_plural = _("Правила для команд")

    def __str__(self):
        return f'{self.team_id} {self.rule}'
