from typing import Union

from django.conf import settings
from django.core.cache import caches
from django.db import models, transaction
from django.dispatch import receiver
from django.utils import timezone

from lms.courses.models import Course, CourseGroup
from lms.enrollments.models import EnrolledUser
from lms.staff.models import StaffProfile

from .models import CourseFollower, MailingEvent
from .services import sync_course_mailings, unsubscribe_on_create_enrolled_user, unsubscribe_on_dismiss
from .settings import MAILING_TASK_COURSE_NEW_FOLLOWER
from .tasks import (
    mailing_course_available_again_task, mailing_course_enroll_begin_task, mailing_course_new_follower_task,
    process_mailing_event_task,
)


def mail_on_course_group_available_again(course_group: CourseGroup):
    key = f'course_enroll_{course_group.course_id}'

    delta_participants = max(0, course_group.max_participants - course_group.num_participants)
    if course_group.become_available_again and delta_participants > 0:
        def mail_users_available_again():
            if not caches['default'].get(key):
                mailing_course_available_again_task.apply_async(
                    kwargs={
                        'course_id': course_group.course_id,
                        'group_id': course_group.id,
                        'delta_participants': delta_participants,
                    },
                    countdown=settings.MAIL_COURSE_AVAILABLE_AGAIN_THROTTLE_TIME,
                )
                if settings.COURSE_ENROLL_THROTTLE_TIME:
                    caches['default'].set(key, True, timeout=settings.COURSE_ENROLL_THROTTLE_TIME)

        transaction.on_commit(mail_users_available_again)


def mail_on_change_enroll_begin(instance: Union[Course, CourseGroup], created: bool):
    course_id = instance.id if isinstance(instance, Course) else instance.course_id
    key = f'course_enroll_{course_id}'
    prev_enroll_begin = instance.tracker.previous('enroll_begin') or None
    now_moment = timezone.now()
    if (
        instance.available_for_enroll and
        (
            created or
            (
                instance.tracker.has_changed('enroll_begin') and
                (
                    instance.enroll_begin is None or
                    instance.enroll_begin < now_moment
                ) and
                prev_enroll_begin is not None and
                prev_enroll_begin >= now_moment
            )
        )
    ):
        def mail_users_enroll_begin():
            if not caches['default'].get(key):
                mailing_course_enroll_begin_task.apply_async(
                    kwargs={
                        'course_id': course_id,
                    },
                    countdown=settings.MAIL_COURSE_ENROLL_BEGIN_THROTTLE_TIME,
                )
                if settings.COURSE_ENROLL_THROTTLE_TIME:
                    caches['default'].set(key, True, timeout=settings.COURSE_ENROLL_THROTTLE_TIME)

        transaction.on_commit(mail_users_enroll_begin)


def mail_on_change_course_group(course_group: CourseGroup, created: bool):
    course = course_group.course
    if not course.enable_followers:
        return

    mail_on_course_group_available_again(course_group=course_group)
    mail_on_change_enroll_begin(instance=course_group, created=created)


@receiver(signal=models.signals.post_save, sender=Course)
def course_post_save_handler(instance: Course, **kwargs):
    if instance.tracker.has_changed('enable_followers'):
        sync_course_mailings(course=instance)


@receiver(signal=models.signals.post_save, sender=CourseFollower)
def course_follower_post_save_handler(instance: CourseFollower, **kwargs):
    if (
        instance.tracker.has_changed('is_active') and
        instance.is_active and
        instance.course.mailings.filter(
            mailing__mailing_task=MAILING_TASK_COURSE_NEW_FOLLOWER,
            is_active=True,
            mailing__is_active=True,
        ).exists()
    ):
        key = f'mailing_course_new_follower_task_{instance.id}'
        if not caches['default'].get(key):
            def mail_follower():
                mailing_course_new_follower_task.delay(course_follower_id=instance.id)
                if settings.MAIL_NEW_FOLLOWER_THROTTLE_TIME:
                    caches['default'].set(key, True, timeout=settings.MAIL_NEW_FOLLOWER_THROTTLE_TIME)

            transaction.on_commit(mail_follower)


@receiver(signal=models.signals.post_save, sender=MailingEvent)
def mailing_event_post_save_handler(instance: MailingEvent, created, **kwargs):
    if created:
        def process_mailing_event():
            options = {}
            if instance.scheduled:
                options['eta'] = instance.scheduled
            process_mailing_event_task.apply_async(
                kwargs={
                    'mailing_event_id': instance.id,
                },
                **options
            )
        transaction.on_commit(process_mailing_event)


@receiver(signal=models.signals.post_save, sender=Course)
def course_in_mailing_post_save_handler(instance: Course, created: bool, **kwargs):
    mail_on_change_enroll_begin(instance=instance, created=created)


@receiver(signal=models.signals.post_save, sender=CourseGroup)
def course_group_in_mailing_post_save_handler(instance: CourseGroup, created: bool, **kwargs):
    mail_on_change_course_group(course_group=instance, created=created)


@receiver(signal=models.signals.post_save, sender=EnrolledUser)
def enrolled_user_in_mailing_post_save_handler(instance: EnrolledUser, created: bool, **kwargs):
    if not instance.tracker.has_changed('status'):
        return

    if not instance.course.enable_followers:
        return

    if (
        instance.tracker.previous('status') in EnrolledUser.USER_STATUS_KEYS and
        instance.status not in EnrolledUser.USER_STATUS_KEYS
    ):
        key = f'mailing_course_available_again_task_{instance.course_id}'
        if not caches['default'].get(key):
            def mail_users():
                mailing_course_available_again_task.delay(
                    course_id=instance.course_id,
                    group_id=instance.group_id,
                    delta_participants=1,
                )
                if settings.COURSE_ENROLL_THROTTLE_TIME:
                    caches['default'].set(key, True, timeout=settings.COURSE_ENROLL_THROTTLE_TIME)

            transaction.on_commit(mail_users)

    if created:
        unsubscribe_on_create_enrolled_user(enrolled_user=instance)


@receiver(signal=models.signals.post_save, sender=StaffProfile)
def staff_profile_in_mailing_post_save_handler(instance: StaffProfile, **kwargs):
    if instance.tracker.has_changed('is_dismissed') and instance.is_dismissed:
        unsubscribe_on_dismiss(staff_profile=instance)
