from django_filters.rest_framework import DjangoFilterBackend
from drf_spectacular.utils import OpenApiParameter, extend_schema

from django.core.exceptions import ValidationError as DjangoValidationError
from django.utils import timezone
from django.utils.translation import gettext

from rest_framework import status
from rest_framework.decorators import action
from rest_framework.exceptions import ValidationError
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.serializers import as_serializer_error

from lms.core.views.viewsets import APIModelViewSet
from lms.courses.models import Course

from ..models import CourseFollower, CourseMailing
from ..serializers import CourseMailingListSerializer, MyCourseFollowsListSerializer


class CourseMailingViewSet(APIModelViewSet):
    serializer_class = CourseMailingListSerializer
    serializer_classes = {
        'list': CourseMailingListSerializer,
    }
    queryset = CourseMailing.objects.active().select_related('course', 'mailing')
    pagination_class = None
    permission_classes = [
        IsAuthenticated,
    ]
    filter_backends = [DjangoFilterBackend]

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            filter_kwargs = {'course_id': course_id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset

    @extend_schema(
        summary=gettext("Список рассылок курса"),
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)


class CourseFollowViewSet(APIModelViewSet):
    """
    Подписка на курс
    """
    queryset = Course.objects.all()
    permission_classes = [
        IsAuthenticated,
    ]

    def get_course(self):
        if not hasattr(self, '_course'):
            course_id = self.kwargs['pk']
            user = self.request.user
            qs = Course.objects.active(user=user)
            self._course = get_object_or_404(qs, pk=course_id)

        return self._course

    @extend_schema(
        request=None,
        responses={204: None},
        summary=gettext("Подписка на курс")
    )
    @action(detail=True, methods='post')
    def follow(self, request, *args, **kwargs):
        try:
            instance, _ = CourseFollower.objects.update_or_create(
                course=self.get_course(),
                user=self.request.user,
                defaults={
                    'is_active': True,
                    'subscribed_date': timezone.now(),
                    'unsubscribed_date': None,
                    'unsubscription_reason': None,
                }
            )
        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))

        return Response(status=status.HTTP_204_NO_CONTENT)

    @extend_schema(
        request=None,
        responses={204: None},
        summary=gettext("Отписка от курса")
    )
    @action(detail=True, methods='post')
    def unfollow(self, request, *args, **kwargs):
        try:
            instance = CourseFollower.objects.filter(
                course=self.get_course(),
                user=self.request.user,
            ).first()
            if instance and instance.is_active:
                instance.is_active = False
                instance.unsubscription_reason = CourseFollower.UnsubscriptionReasonChoice.USER
                if instance.unsubscribed_date is None:
                    instance.unsubscribed_date = timezone.now()
                instance.save()

        except DjangoValidationError as exc:
            raise ValidationError(detail=as_serializer_error(exc))

        return Response(status=status.HTTP_204_NO_CONTENT)


class MyCourseFollowsListViewSet(APIModelViewSet):
    serializer_class = MyCourseFollowsListSerializer
    serializer_classes = {
        'list': MyCourseFollowsListSerializer,
    }
    queryset = CourseFollower.objects.active().filter(is_active=True).select_related('course')
    pagination_class = None
    permission_classes = [
        IsAuthenticated,
    ]
    filter_backends = [DjangoFilterBackend]
    filterset_fields = ['course_id']

    def get_queryset(self):
        if self.request.user and self.request.user.is_authenticated:
            qs = super().get_queryset()
            return qs.filter(user=self.request.user)

        return CourseFollower.objects.none()

    @extend_schema(
        summary=gettext("курсы, на которые подписан пользователь"),
        parameters=[
            OpenApiParameter(name='course_id', required=False, type=int),
        ]
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)
