import logging

from model_utils.models import TimeStampedModel

from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError
from django.db import models
from django.utils.translation import gettext_lazy as _

from lms.core.models.mixins import ActiveModelMixin
from lms.courses.models import Course

from .querysets import MentorshipQuerySet

logger = logging.getLogger(__name__)

User = get_user_model()


class Mentorship(ActiveModelMixin, TimeStampedModel):
    mentor = models.ForeignKey(
        User,
        verbose_name=_("наставник"),
        related_name='mentorships_as_mentor',
        on_delete=models.PROTECT,
    )
    mentee = models.ForeignKey(
        User,
        verbose_name=_("подопечный"),
        related_name='mentorships_as_mentee',
        on_delete=models.PROTECT,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='mentorships',
        null=True,
        blank=True,
        on_delete=models.CASCADE,
    )
    created_by = models.ForeignKey(
        User,
        verbose_name=_("создано"),
        related_name='created_mentorships',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    due_date = models.DateTimeField(
        _("дата завершения"),
        null=True,
        blank=True,
    )

    class Meta:
        verbose_name = _("наставничество")
        verbose_name_plural = _("наставничества")

    objects = MentorshipQuerySet.as_manager()

    def clean(self):
        if self.mentor_id == self.mentee_id:
            raise ValidationError(
                _("наставник не должен совпадать с подопечным"),
                code='invalid',
            )

    def save(self, *args, **kwargs):
        self.full_clean()
        super().save(*args, **kwargs)

    def __str__(self):
        return f'Mentorship: {self.id} Mentor: {self.mentor_id} Mentee: {self.mentee_id}'
