from django.contrib.auth import get_user_model

from rest_framework import serializers

from lms.courses.models import Course, CourseStudent
from lms.staff.models import StaffProfile

from .models import Mentorship

User = get_user_model()


# API Serializers
# ===============
class StaffProfileInlineSerializer(serializers.ModelSerializer):
    class Meta:
        model = StaffProfile
        fields = (
            'joined_at',
        )
        read_only_fields = fields


class UserInlineSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = (
            'id', 'username', 'first_name', 'last_name',
        )
        read_only_fields = fields


class CourseStudentProgressScoreField(serializers.IntegerField):
    def to_representation(self, course_student: CourseStudent):
        progress = next(iter(course_student.course_progresses.all()), None)
        if not progress:
            return None
        return progress.score


class CourseStudentsStartDateField(serializers.IntegerField):
    def to_representation(self, course_student: CourseStudent):
        progress = next(iter(course_student.course_progresses.all()), None)
        if not progress:
            return None
        return progress.created


class CourseStudentInlineSerializer(serializers.ModelSerializer):
    score = CourseStudentProgressScoreField(source='*', allow_null=True)

    class Meta:
        model = CourseStudent
        fields = (
            'id', 'status', 'is_passed', 'score',
        )
        read_only_fields = fields


class CourseInlineSerializer(serializers.ModelSerializer):
    type = serializers.ChoiceField(choices=Course.TypeChoices.choices, source='course_type')

    class Meta:
        model = Course
        fields = (
            'id', 'name', 'type',
        )
        read_only_fields = fields


class CourseStudentInlineDetailSerializer(serializers.ModelSerializer):
    start_date = CourseStudentsStartDateField(source='*', allow_null=True)
    score = CourseStudentProgressScoreField(source='*', allow_null=True)
    course = CourseInlineSerializer(read_only=True)

    class Meta:
        model = CourseStudent
        fields = (
            'id', 'status', 'start_date', 'score', 'is_passed', 'course',
        )
        read_only_fields = fields


class MentorshipInlineSerializer(serializers.ModelSerializer):
    mentor = UserInlineSerializer()

    class Meta:
        model = Mentorship
        fields = (
            'id', 'due_date', 'mentor',
        )
        read_only_fields = fields


class UserColleagueListSerializer(serializers.ModelSerializer):
    """
    Список коллег пользователя (подчинённые и подопечные)
    """
    profile = StaffProfileInlineSerializer(source='staffprofile')
    students = CourseStudentInlineSerializer(many=True, source='in_courses')
    mentorships = MentorshipInlineSerializer(many=True, source='mentorships_as_mentee')

    class Meta:
        model = User
        fields = (
            'id', 'username', 'first_name', 'last_name',
            'profile', 'students', 'mentorships',
        )
        read_only_fields = fields


class UserColleagueDetailSerializer(serializers.ModelSerializer):
    """
    Информация о коллеге пользователя (подчинённом или подопечном)
    """
    students = CourseStudentInlineDetailSerializer(many=True, source='in_courses')

    class Meta:
        model = User
        fields = (
            'id', 'username', 'first_name', 'last_name', 'students',
        )
        read_only_fields = fields
