import sys

from django.apps import apps as global_apps
from django.core.exceptions import ImproperlyConfigured
from django.db import DEFAULT_DB_ALIAS, migrations, router, transaction
from django.db.utils import IntegrityError

from ..settings import MODULETYPES_EXCLUDE_MODULES, MODULETYPES_MODULE_CLASS


class RenameModuleType(migrations.RunPython):
    def __init__(self, app_label, old_model, new_model):
        self.app_label = app_label
        self.old_model = old_model
        self.new_model = new_model
        super().__init__(self.rename_forward, self.rename_backward)

    def _rename(self, apps, schema_editor, old_model, new_model):
        ModuleType = apps.get_model('moduletypes', 'ModuleType')  # noqa: N806
        db = schema_editor.connection.alias
        if not router.allow_migrate_model(db, ModuleType):
            return

        try:
            module_type = ModuleType.objects.db_manager(db).get_by_natural_key(self.app_label, old_model)
        except ModuleType.DoesNotExist:
            pass
        else:
            module_type.model = new_model

            try:
                with transaction.atomic(using=db):
                    module_type.save(update_fields={'model'})
            except IntegrityError:
                # Gracefully fallback if a stale module type causes a
                # conflict as remove_stale_moduletypes will take care of
                # asking the user what should be done next.
                module_type.model = old_model
            else:
                # Clear the cache as the `get_by_natual_key()` call will cache
                # the renamed ModuleType instance by its old model name.
                ModuleType.objects.clear_cache()

    def rename_forward(self, apps, schema_editor):
        self._rename(apps, schema_editor, self.old_model, self.new_model)

    def rename_backward(self, apps, schema_editor):
        self._rename(apps, schema_editor, self.new_model, self.old_model)


def inject_rename_moduletypes_operations(plan=None, apps=global_apps, using=DEFAULT_DB_ALIAS, **kwargs):
    """
    Вставляет RenameModuleType операции после каждой запланированной RenameModel,
    чтобы при переменовании моделей, которые являются ModuleType, менять имя в таблице ModuleType
    """
    if plan is None:
        return

    try:
        ModuleType = apps.get_model('moduletypes', 'ModuleType')  # noqa: N806
    except LookupError:
        available = False
    else:
        if not router.allow_migrate_model(using, ModuleType):
            return
        available = True

    for migration, backward in plan:
        if (migration.app_label, migration.name) == ('moduletypes', '0001_initial'):
            # Тут нечего делать, если еще не применилась миграция moduletypes
            if backward:
                break
            else:
                available = True
                continue

        # модель ModuleType еще не доступна
        if not available:
            continue

        inserts = []
        for index, operation in enumerate(migration.operations):
            # TODO: фильтровать только те модели, которые наследованы от CourseModule
            if isinstance(operation, migrations.RenameModel):
                operation = RenameModuleType(
                    migration.app_label, operation.old_name_lower, operation.new_name_lower
                )
                inserts.append((index + 1, operation))
        for inserted, (index, operation) in enumerate(inserts):
            migration.operations.insert(inserted + index, operation)


def get_moduletypes_and_models(app_config, using, ModuleType, Module, excluded=None):  # noqa: N803
    if not router.allow_migrate_model(using, ModuleType):
        return None, None

    ModuleType.objects.clear_cache()

    module_types = {
        mt.model: mt
        for mt in ModuleType.objects.using(using).filter(app_label=app_config.label)
    }

    app_models = {}

    for model in app_config.get_models():
        if excluded and model in excluded:
            continue

        if not issubclass(model, Module) or model is Module:
            continue

        app_models[model._meta.model_name] = model

    return module_types, app_models


def get_module_model(apps):
    try:
        return apps.get_model(MODULETYPES_MODULE_CLASS)
    except LookupError:
        raise ImproperlyConfigured(
            f"MODULETYPES_MODULE_CLASS refers to model '{MODULETYPES_MODULE_CLASS}' that has not been installed"
        )


def create_module_types(app_config, verbosity=2, interactive=True, using=DEFAULT_DB_ALIAS, apps=global_apps, **kwargs):
    """
    Создает типы модулей, наследующих CourseModule
    """
    if not app_config.models_module:
        return

    app_label = app_config.label
    try:
        app_config = apps.get_app_config(app_label)
        ModuleType = apps.get_model('moduletypes', 'ModuleType')  # noqa: N806
        excluded = [apps.get_model(model) for model in MODULETYPES_EXCLUDE_MODULES]
    except LookupError:
        return

    # ModuleType.objects.clear_cache()
    Module = get_module_model(apps)  # noqa: N806
    module_types, app_models = get_moduletypes_and_models(app_config, using, ModuleType, Module, excluded)

    if not app_models:
        return

    mts = [
        ModuleType(
            app_label=app_label,
            model=model_name,
        )
        for (model_name, model) in app_models.items()
        if model_name not in module_types
    ]
    ModuleType.objects.using(using).bulk_create(mts)

    if verbosity >= 2:
        for mt in mts:
            sys.stdout.write(f"Adding module type '{mt.app_label} | {mt.model}\n")
