from django.apps import apps
from django.contrib.contenttypes.models import ContentTypeManager
from django.db import models
from django.utils.translation import gettext_lazy as _


class ModuleTypeManager(ContentTypeManager):
    pass


class ModuleType(models.Model):
    app_label = models.CharField(max_length=100)
    model = models.CharField(_('python model class name'), max_length=100)
    is_active = models.BooleanField(_("активен"), default=True)
    objects = ModuleTypeManager()

    class Meta:
        unique_together = (('app_label', 'model'),)
        verbose_name = _('тип модуля')
        verbose_name_plural = _('типы модулей')

    def __str__(self):
        return self.name

    @property
    def name(self) -> str:
        model = self.model_class()
        if not model:
            return self.model
        return str(model._meta.verbose_name)

    def model_class(self):
        try:
            return apps.get_model(self.app_label, self.model)
        except LookupError:
            return None

    def get_object_for_this_type(self, **kwargs):
        return self.model_class()._base_manager.using(self._state.db).get(**kwargs)

    def get_all_objects_for_this_type(self, **kwargs):
        return self.model_class()._base_manager.using(self._state.db).filter(**kwargs)

    def natural_key(self):
        return self.app_label, self.model


class Module(models.Model):
    module_type = models.ForeignKey(
        ModuleType,
        verbose_name=_("тип модуля"),
        related_name='modules',
        on_delete=models.PROTECT,
        editable=False,
    )

    DETECT_MODULE_TYPE = True

    class Meta:
        abstract = True

    def get_module_type(self):
        return ModuleType.objects.get_for_model(self)

    def save(self, *args, **kwargs):
        if self.DETECT_MODULE_TYPE and not hasattr(self, 'module_type'):
            self.module_type = self.get_module_type()
        super().save(*args, **kwargs)
