import logging
import tempfile
from typing import Iterable, TextIO, Tuple
from uuid import uuid4

from django.core.files import File
from django.db import transaction

from .models import ReportFile

log = logging.getLogger(__name__)


class ReportFileLockedOrDoesNotExist(Exception):
    pass


def report_to_file(colnames: Iterable[str], array: Iterable[Iterable], output_type: str = 'xlsx') -> Tuple[TextIO, str]:
    from pyexcel import free_resources, get_sheet

    f = tempfile.NamedTemporaryFile('wb+')
    sheet = get_sheet(array=array)
    sheet.colnames = colnames
    sheet.save_as(
        f.name,
        force_file_type=output_type,
    )
    free_resources()
    f.flush()
    return f, f'{uuid4().hex}.{output_type}'


@transaction.atomic()
def create_report(
    base_report: ReportFile,
    force=False,
):

    if base_report.status != ReportFile.StatusChoices.CREATED and not force:
        return

    colnames, array = base_report.get_report_data()
    f, filename = report_to_file(colnames, array)

    base_report.status = ReportFile.StatusChoices.COMPLETED
    base_report.file = File(f, name=filename)
    base_report.error_message = ''
    base_report.save(update_fields=['status', 'file', 'error_message'])
