import logging
import traceback
from uuid import UUID

from django.db import transaction
from django.utils.translation import gettext_lazy as _

from config import celery_app as app
from lms.reports.services import create_report

from .models import ReportFile

log = logging.getLogger(__name__)


@app.task(bind=True)
def run_report_task(self, report_id: UUID, force: bool = False):
    with transaction.atomic():
        base_report = None
        try:
            base_report = ReportFile.objects.filter(id=report_id).select_for_update(skip_locked=True).first()
            if base_report is None:
                log.error(_("ReportFile %s locked or does not exist") % report_id)
                return

            create_report(base_report=base_report, force=force)

        except Exception:  # по непонятным проблемам пишем ошибку в лог и в объект
            exc_info = traceback.format_exc()
            log.error(exc_info)
            if base_report is not None:
                base_report.error_message = exc_info
                base_report.status = ReportFile.StatusChoices.ERROR
                base_report.save(update_fields=['error_message', 'status'])


@app.task(bind=True)
def run_report_periodic_task(self, *args, **kwargs):
    reports = ReportFile.objects.filter(status=ReportFile.StatusChoices.CREATED)
    for report in reports:
        run_report_task.delay(report_id=report.id)
