from drf_spectacular.utils import extend_schema

from django.utils.translation import gettext

from rest_framework import status
from rest_framework.generics import get_object_or_404
from rest_framework.response import Response

from lms.core.views.viewsets import LabModelViewSet
from lms.courses.models import Course
from lms.courses.permissions import CourseObjectPermission

from ..models import EnrolledUsersReport, ReportFile, StudentSlotsReport
from ..serializers import (
    EnrolledUsersReportDetailSerializer, ReportFileDetailLabSerializer, StudentSlotsReportDetailSerializer,
)


class ReportFileLabViewSet(LabModelViewSet):
    serializer_class = ReportFileDetailLabSerializer
    serializer_classes = {
        'retrieve': ReportFileDetailLabSerializer
    }

    queryset = ReportFile.objects.all()

    @extend_schema(
        summary=gettext("Получить отчет")
    )
    def retrieve(self, *args, **kwargs):
        return super().retrieve(*args, **kwargs)


class EnrolledUsersReportLabViewSet(LabModelViewSet):
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]
    serializer_class = EnrolledUsersReportDetailSerializer
    serializer_classes = {
        'retrieve': EnrolledUsersReportDetailSerializer
    }

    queryset = EnrolledUsersReport.objects.all()

    def get_course(self, obj=None):
        if not hasattr(self, '_course'):
            if obj is not None:
                self._course = obj.course
            if self.action == 'create':
                self._course = get_object_or_404(Course.objects.all(), pk=self.kwargs.get('pk'))

        return self._course

    @extend_schema(
        summary=gettext("Запустить создание отчета: заявки на курс")
    )
    def create(self, *args, **kwargs):
        report = EnrolledUsersReport.objects.create(course=self.get_course())
        response_serializer = self.get_retrieve_serializer(report)
        headers = self.get_success_headers(response_serializer.data)
        return Response(response_serializer.data, status=status.HTTP_202_ACCEPTED, headers=headers)


class StudentSlotsReportLabViewSet(LabModelViewSet):
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]
    serializer_class = StudentSlotsReportDetailSerializer
    serializer_classes = {
        'retrieve': StudentSlotsReportDetailSerializer
    }

    queryset = StudentSlotsReport.objects.all()

    def get_course(self, obj=None):
        if not hasattr(self, '_course'):
            if obj is not None:
                self._course = obj.course
            if self.action == 'create':
                self._course = get_object_or_404(Course.objects.all(), pk=self.kwargs.get('pk'))

        return self._course

    @extend_schema(
        summary=gettext("Запустить создание отчета: заявки на слоты курса")
    )
    def create(self, *args, **kwargs):
        report = StudentSlotsReport.objects.create(course=self.get_course())
        response_serializer = self.get_retrieve_serializer(report)
        headers = self.get_success_headers(response_serializer.data)
        return Response(response_serializer.data, status=status.HTTP_202_ACCEPTED, headers=headers)
