from typing import Union

from django.db import transaction
from django.db.models.signals import post_delete, post_save

from lms.courses.services import refresh_course_module_weight_sum_cache

from .models import LinkResource, TextResource, VideoResource


def resource_post_save_handler(instance: Union[TextResource, LinkResource, VideoResource], created: bool, **kwargs):
    if created or instance.field_tracker.has_changed('weight') or instance.field_tracker.has_changed('is_active'):
        transaction.on_commit(lambda: refresh_course_module_weight_sum_cache(course_id=instance.course_id))


def resource_post_delete_handler(instance: Union[TextResource, LinkResource, VideoResource], **kwargs):
    transaction.on_commit(lambda: refresh_course_module_weight_sum_cache(course_id=instance.course_id))


for model in (TextResource, LinkResource, VideoResource):
    post_save.connect(resource_post_save_handler, sender=model)
    post_delete.connect(resource_post_delete_handler, sender=model)
