from django_admin_listfilter_dropdown.filters import RelatedOnlyDropdownFilter
from django_object_actions import DjangoObjectActions, takes_instance_or_queryset
from simple_history.admin import SimpleHistoryAdmin

from django.contrib import admin, messages
from django.utils.translation import gettext_lazy as _

from .models import Scorm, ScormFile, ScormResource, ScormResourceStudentAttempt, ScormStudentAttempt
from .tasks import prepare_scorm_task


class ScormResourceInlineAdmin(admin.TabularInline):
    model = ScormResource
    fields = ('id', 'scorm_file', 'resource_id', 'href')
    readonly_fields = fields
    extra = 0
    can_delete = False

    def has_add_permission(self, request, obj=None):
        return False


@admin.register(ScormResource)
class ScormResourceAdmin(admin.ModelAdmin):
    fields = ('id', 'scorm_file', 'resource_id', 'href')
    raw_id_fields = ('scorm_file',)
    list_select_related = ('scorm_file',)
    readonly_fields = fields

    def has_add_permission(self, request, obj=None):
        return False


@admin.register(ScormFile)
class ScormFileAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ('scorm', 'scorm_status')
    list_filter = ('scorm_status',)
    raw_id_fields = ('course_file',)
    readonly_fields = ('manifest', 'public_url')

    inlines = [
        ScormResourceInlineAdmin,
    ]

    actions = [
        'prepare_scorm_action',
    ]

    change_actions = [
        'prepare_scorm_action',
    ]

    @takes_instance_or_queryset
    def prepare_scorm_action(self, request, queryset):
        for scorm_file in queryset:
            try:
                prepare_scorm_task.delay(scorm_file_id=scorm_file.id, force=True)
                message = _(f"запуск подготовки scorm={scorm_file.id}")
                level = messages.INFO
            except Exception as ex:
                message = str(ex)
                level = messages.ERROR
            self.message_user(
                request=request,
                message=message,
                level=level,
            )

    prepare_scorm_action.short_description = _("подготовка scorm")
    prepare_scorm_action.label = _("подготовка scorm")


class ScormFileInlineAdmin(admin.TabularInline):
    model = ScormFile
    fields = ('id', 'public_url', 'scorm_status')
    readonly_fields = fields
    extra = 0
    can_delete = False

    def has_add_permission(self, request, obj=None):
        return False


@admin.register(Scorm)
class ScormAdmin(admin.ModelAdmin):
    list_display = ('course', 'name', 'is_active', 'created', 'modified')
    list_filter = (
        ('course', RelatedOnlyDropdownFilter),
    )
    list_select_related = ('course',)
    raw_id_fields = ('course', 'current_file')
    search_fields = ('name',)

    inlines = [
        ScormFileInlineAdmin,
    ]


@admin.register(ScormStudentAttempt)
class ScormStudentAttemptAdmin(SimpleHistoryAdmin):
    list_display = ('scorm', 'student', 'current_attempt')
    list_filter = (
        ('scorm__course', RelatedOnlyDropdownFilter),
        ('scorm', RelatedOnlyDropdownFilter),
        ('scorm_file', RelatedOnlyDropdownFilter),
        ('student', RelatedOnlyDropdownFilter),
    )
    list_select_related = ('scorm', 'scorm__course', 'student')
    raw_id_fields = ('scorm', 'student')
    readonly_fields = ('scorm',)


@admin.register(ScormResourceStudentAttempt)
class ScormResourceStudentAttemptAdmin(admin.ModelAdmin):
    list_display = ('current_attempt', 'student', 'scorm_resource')
    list_filter = (
        ('student__course', RelatedOnlyDropdownFilter),
        ('student__user', RelatedOnlyDropdownFilter),
        ('scorm_resource__scorm_file', RelatedOnlyDropdownFilter),
        ('scorm_resource', RelatedOnlyDropdownFilter),
    )
    list_select_related = (
        'student',
        'student__course',
        'student__user',
        'scorm_resource',
        'scorm_resource__scorm_file',
    )
    raw_id_fields = ('student', 'scorm_resource')
