from django.db import transaction
from django.utils.translation import gettext_lazy as _

from rest_framework import serializers

from lms.courses.models import CourseFile
from lms.courses.serializers import (
    CourseModuleCreateBaseSerializer, CourseModuleDetailBaseSerializer, CourseModuleUpdateBaseSerializer,
)

from .models import Scorm, ScormFile, ScormResource, ScormResourceStudentAttempt, ScormStudentAttempt


# LABAPI
# ===
class ScormResourceListInlineLabSerializer(serializers.ModelSerializer):
    class Meta:
        model = ScormResource
        fields = (
            'id', 'resource_id', 'href',
            'created', 'modified',
        )
        read_only_fields = fields


class ScormFileCreateLabSerializer(serializers.ModelSerializer):
    scorm_id = serializers.PrimaryKeyRelatedField(
        source='scorm', queryset=Scorm.objects.all(),
    )
    course_file_id = serializers.PrimaryKeyRelatedField(
        source='course_file', queryset=CourseFile.objects.all(),
    )

    class Meta:
        model = ScormFile
        fields = (
            'scorm_id', 'course_file_id', 'comment'
        )


class ScormFileDetailLabSerializer(serializers.ModelSerializer):
    resources = ScormResourceListInlineLabSerializer(many=True)
    filename = serializers.SerializerMethodField()

    def get_filename(self, obj: ScormFile) -> str:
        return obj.course_file.filename

    class Meta:
        model = ScormFile
        fields = (
            "id", "filename", "public_url",
            "scorm_status", "comment", "error_messages",
            "manifest", "resources", "created"
        )
        read_only_fields = fields


class ScormFileListLabSerializer(serializers.ModelSerializer):
    resources = ScormResourceListInlineLabSerializer(many=True)
    filename = serializers.SerializerMethodField()

    def get_filename(self, obj: ScormFile) -> str:
        return obj.course_file.filename

    class Meta:
        model = ScormFile
        fields = (
            "id", "filename", "public_url",
            "scorm_status", "comment", "error_messages",
            "manifest", "resources", "created"
        )
        read_only_fields = fields


class ScormDetailLabSerializer(CourseModuleDetailBaseSerializer):
    current_file = ScormFileDetailLabSerializer(required=False)

    class Meta(CourseModuleDetailBaseSerializer.Meta):
        model = Scorm
        fields = CourseModuleDetailBaseSerializer.Meta.fields + (
            'course_id',
            'max_attempts',
            'current_file',
        )
        read_only_fields = fields


class ScormCreateLabSerializer(CourseModuleCreateBaseSerializer):
    course_file_id = serializers.PrimaryKeyRelatedField(
        queryset=CourseFile.objects.values_list('id', flat=True), required=False
    )
    comment = serializers.CharField(required=False)

    class Meta(CourseModuleCreateBaseSerializer.Meta):
        model = Scorm
        fields = CourseModuleCreateBaseSerializer.Meta.fields + (
            'course_file_id', 'max_attempts', 'comment'
        )

    @transaction.atomic
    def create(self, validated_data):
        course_file_id = validated_data.pop('course_file_id', None)
        comment = validated_data.pop('comment', '')
        instance: Scorm = super().create(validated_data)
        if course_file_id:
            course_file = CourseFile.objects.filter(pk=course_file_id).first()
            ScormFile.objects.create(
                course_file=course_file,
                comment=comment,
                scorm=instance,
            )
        return instance


class ScormUpdateLabSerializer(CourseModuleUpdateBaseSerializer):
    class Meta(CourseModuleUpdateBaseSerializer.Meta):
        model = Scorm
        fields = CourseModuleUpdateBaseSerializer.Meta.fields + (
            'max_attempts',
        )


# API
# ===
class ScormResourceListInlineSerializer(serializers.ModelSerializer):
    class Meta:
        model = ScormResource
        fields = (
            'id', 'resource_id', 'href',
            'created', 'modified',
        )
        read_only_fields = fields


class ScormFileDetailInlineSerializer(serializers.ModelSerializer):
    class Meta:
        model = ScormFile
        fields = (
            'id', 'comment',
            'created', 'modified',
        )
        read_only_fields = fields


class ScormDetailSerializer(serializers.ModelSerializer):
    current_file = ScormFileDetailInlineSerializer(allow_null=True, read_only=True, label=_("Текущая версия файла"))

    class Meta:
        model = Scorm
        fields = (
            'id', 'name', 'description', 'max_attempts', 'is_active',
            'current_file',
            'created', 'modified',
        )
        read_only_fields = fields


class ScormFileDetailWithResourcesInlineSerializer(serializers.ModelSerializer):
    resources = ScormResourceListInlineSerializer(many=True, read_only=True, label=_("SCORM-ресурсы"))

    class Meta:
        model = ScormFile
        fields = (
            'id', 'public_url', 'comment', 'resources',
            'created', 'modified',
        )
        read_only_fields = fields


class ScormStudentAttemptDetailSerializer(serializers.ModelSerializer):
    scorm_file = ScormFileDetailWithResourcesInlineSerializer(read_only=True, label=_("SCORM-файл"))

    class Meta:
        model = ScormStudentAttempt
        fields = (
            'current_attempt', 'scorm_file',
        )
        read_only_fields = fields


class ScormStudentAttemptCreateSerializer(serializers.ModelSerializer):
    """
    Создание попытки
    """
    scorm_id = serializers.PrimaryKeyRelatedField(
        source='scorm', queryset=Scorm.objects.active().only('id'), required=True
    )

    class Meta:
        model = ScormStudentAttempt
        fields = (
            'scorm_id',
        )


class ScormResourceStudentAttemptDetailSerializer(serializers.ModelSerializer):
    """
    Данные пользователя по попытке прохождения SCORM-ресурса
    """

    class Meta:
        model = ScormResourceStudentAttempt
        fields = (
            'current_attempt',
            'data',
        )
        read_only_fields = fields


class ScormResourceStudentAttemptUpdateSerializer(serializers.ModelSerializer):
    """
    Обновление данных пользователя по попытке прохождения SCORM-ресурса
    """

    class Meta:
        model = ScormResourceStudentAttempt
        fields = (
            'data',
        )
