import logging
import os.path
import tempfile
import zipfile
from decimal import ROUND_HALF_UP, Decimal, InvalidOperation
from urllib.parse import urlparse
from uuid import uuid4

import jsonschema
import xmltodict

from django.conf import settings
from django.db import transaction

from lms.contrib.s3.s3_boto_client import s3_boto_client
from lms.courses.models import CourseFile, CourseStudent

from .default_init_state import DEFAULT_INIT_STATE
from .manifest_schema import MANIFEST_SCHEMA
from .models import Scorm, ScormFile, ScormResource, ScormResourceStudentAttempt, ScormStudentAttempt
from .settings import SCORM_MANIFEST_FILENAME

log = logging.getLogger(__name__)


class ScormValidationError(Exception):
    pass


def safe_filename(filename):
    base_filename, extension = os.path.splitext(filename)
    return f'_{base_filename}_{uuid4().hex}{extension}'


def normalize_item(item):
    if not item or not isinstance(item, dict):
        return
    if 'item' in item:
        subitems = item['item']
        if isinstance(subitems, dict) and 'item' in subitems:
            subitems = [subitems]
        if isinstance(subitems, list):
            del item['item']
            item['items'] = subitems
            for subitem in subitems:
                normalize_item(subitem)


def normalize_manifest(manifest):
    resources = manifest["manifest"]["resources"]["resource"]
    if isinstance(resources, dict):
        resources = [resources]
    manifest["manifest"]["resources"]["resource"] = resources

    organizations = manifest["manifest"]["organizations"]["organization"]
    if isinstance(organizations, dict):
        organizations = [organizations]
    manifest["manifest"]["organizations"]["organization"] = organizations

    for organization in manifest["manifest"]["organizations"]["organization"]:
        normalize_item(organization)


def walk_items(item):
    if isinstance(item, dict):
        if 'item' in item:
            yield item['item']
        if 'items' in item:
            for subitem in item['items']:
                yield from walk_items(subitem)


def scorms_upload_destination():
    uuid = uuid4().hex
    return f"{settings.S3_SCORM_BASE_PATH}/{uuid[0]}/{uuid[:2]}/{uuid}"


def prepare_scorm(scorm_file_id: int, force: bool = False) -> None:
    """
    Подготовка scorm-курса к использованию:
    1. Скачивание zip-архива со scorm-курсом с s3
    2. Распаковка архива
    3. Парсинг и валидация манифеста
    4. Заливка распакованного архива со scorm-курсом на s3
    5. Создание ScormResource
    6. Запись манифеста в Scorm
    :param scorm_file_id: идентификатор ScormFile
    :param force: повторная обработка
    """

    with transaction.atomic():
        # Находим scorm
        scorm_file = ScormFile.objects.select_for_update().get(id=scorm_file_id)

        if not scorm_file or (not force and scorm_file.scorm_status != ScormFile.SCORM_MODULE_STATUS_PENDING):
            return

        # Создаем временные папки для скачаннного и распакованнного zip-архива со scorm-курсом
        with tempfile.TemporaryDirectory() as tmp_zip_folder:
            with tempfile.TemporaryDirectory() as tmp_data_folder:
                local_zip_folder = tmp_zip_folder  # временная папка для скачивания zip
                local_data_folder = tmp_data_folder  # временная папка для распаковки zip
                zip_s3_full_path = urlparse(scorm_file.course_file.file).path  # полный путь к zip (s3)
                zip_filename = os.path.basename(zip_s3_full_path)
                safe_zip_filename = safe_filename(zip_filename)
                safe_foldername = os.path.splitext(safe_zip_filename)[0]
                zip_local_full_path = os.path.join(local_zip_folder, zip_filename)  # путь к zip (local)
                local_data_path = os.path.join(local_data_folder, safe_foldername)  # путь к распакованному zip (local)
                s3_data_path = scorms_upload_destination()  # путь к распакованному zip (s3)
                # Скачиваем архив со scorm-курсом
                if not s3_boto_client.file_exists(zip_s3_full_path):
                    scorm_file.course_file.status = CourseFile.CourseFileStatus.ERROR
                    scorm_file.course_file.save(update_fields=["status", "modified"])
                    raise ScormValidationError("File %s not found in s3" % zip_s3_full_path)

                scorm_file.course_file.status = CourseFile.CourseFileStatus.SUCCESS
                scorm_file.course_file.save(update_fields=["status", "modified"])

                s3_boto_client.download_file(
                    s3_file_path=zip_s3_full_path,
                    local_file_path=zip_local_full_path,
                )

                # Распаковываем архив
                with zipfile.ZipFile(zip_local_full_path, 'r') as zip_stream:
                    zip_stream.extractall(local_data_path)

                # Локальный путь к манифесту
                xml_manifest_path = os.path.join(local_data_path, SCORM_MANIFEST_FILENAME)

                if not os.path.exists(xml_manifest_path):
                    raise ScormValidationError('File %s not found' % SCORM_MANIFEST_FILENAME)

                # Открываем манифест и парсим его в json
                with open(xml_manifest_path, 'r') as xml_manifest_file:
                    xml_manifest = xml_manifest_file.read()
                    json_manifest = xmltodict.parse(xml_manifest)

                # нормализация json-манифест
                normalize_manifest(json_manifest)

                # Валидируем json-манифест по схеме
                jsonschema.validate(json_manifest, MANIFEST_SCHEMA)

                # Проверяем, что в ресурсах нет дубликатов
                # И для каждого ресурса существует файл с точкой входа
                # Создаем объекты в памяти ScormResource для sco-ресурсов
                resources_ids = set()
                scorm_resources = []
                resources = json_manifest["manifest"]["resources"]["resource"]
                for resource in resources:
                    scorm_type = resource.get('@adlcp:scormType', 'sco')
                    if scorm_type == 'asset':
                        continue
                    resource_id = resource['@identifier']
                    if resource_id in resources_ids:
                        raise ScormValidationError("Resource id %s is not unique" % resource_id)
                    resources_ids.add(resource_id)

                    href = resource['@href']
                    resource_start_file_path = os.path.join(local_data_path, href)

                    if not os.path.exists(resource_start_file_path):
                        raise ScormValidationError(
                            "Start file %s for resource %s not found" % (
                                href,
                                resource_id,
                            ),
                        )

                    scorm_resources.append(
                        {
                            'scorm_file': scorm_file,
                            'resource_id': resource_id,
                            'href': href,
                        },
                    )

                # Проверяем, что существует дефолтная организация
                # Проверяем, что для каждого листового item существует соответствующий scorm-ресурс
                organizations = json_manifest["manifest"]["organizations"]["organization"]
                default_organization_id = json_manifest["manifest"]["organizations"]["@default"]
                has_default_organization = False
                for organization in organizations:
                    organization_id = organization['@identifier']
                    if organization_id == default_organization_id:
                        has_default_organization = True
                    for leaf_item in walk_items(organization):
                        item_id = leaf_item['@identifier']
                        resource_id = leaf_item['@identifierref']
                        if resource_id not in resources_ids:
                            raise ScormValidationError(
                                'In organization %s in item %s resource %s not found' % (
                                    organization_id,
                                    item_id,
                                    resource_id,
                                ),
                            )

                if not has_default_organization:
                    raise ScormValidationError("No default organization %s" % default_organization_id)

                # Манифест проверен
                # Загружаем распакованный scorm-курс на s3
                s3_boto_client.upload_folder(
                    local_root_path=local_data_path,
                    s3_root_path=s3_data_path,
                )

                # Пишем/обновляем scorm-ресурсы в базу
                for scorm_resource in scorm_resources:
                    ScormResource.objects.update_or_create(
                        scorm_file=scorm_resource['scorm_file'],
                        resource_id=scorm_resource['resource_id'],
                        defaults=scorm_resource,
                    )

                # Записываем json-манифест и ставим отметку, что SCORM готов к использованию
                scorm_file.public_url = s3_data_path
                scorm_file.scorm_status = ScormFile.SCORM_MODULE_STATUS_READY
                scorm_file.manifest = json_manifest
                scorm_file.save()


def create_resource_attempts(attempt: ScormStudentAttempt) -> None:
    resource_attempts_to_create = []
    for resource in attempt.scorm_file.resources.all():
        resource_attempt = ScormResourceStudentAttempt(
            student=attempt.student,
            scorm_resource=resource,
            current_attempt=attempt.current_attempt,
            data=DEFAULT_INIT_STATE,
        )
        resource_attempts_to_create.append(resource_attempt)

    ScormResourceStudentAttempt.objects.bulk_create(
        objs=resource_attempts_to_create,
        batch_size=settings.BULK_BATCH_SIZE_DEFAULT,
    )


def create_new_attempt(student: CourseStudent, scorm: Scorm) -> ScormStudentAttempt:
    attempt = (
        ScormStudentAttempt.objects
        .filter(student=student, scorm=scorm)
        .select_related('student')
        .select_for_update()
        .first()
    )
    if attempt is None:
        attempt = ScormStudentAttempt.objects.create(
            scorm=scorm,
            scorm_file=scorm.current_file,
            student=student,
        )
    else:
        attempt.scorm_file = scorm.current_file
        attempt.save()

    return attempt


def update_scaled_score(attempt: ScormResourceStudentAttempt):
    cmi = attempt.data.get('cmi', {}) if attempt.data else {}
    completion_status = cmi.get('completion_status') or cmi.get('core', {}).get('completion_status')
    success_status = cmi.get('success_status') or cmi.get('core', {}).get('success_status')
    if completion_status == 'completed' and success_status != 'failed':
        scaled_score = 1
    else:
        score = cmi.get('score', {}) or cmi.get('core', {}).get('score', {})
        scaled_score = score.get('scaled')
    if scaled_score is not None:
        try:
            scaled_score = Decimal(scaled_score)
        except (TypeError, InvalidOperation):
            log.warning('Invalid score.scaled=%s found in SCORM state. attempt_id=%s', scaled_score, attempt.id)
            return
        # иногда нам приходили Decimal('NaN'), а is_finite() обрабатывает ещё и случай с Decimal('Infinity')
        if not scaled_score.is_finite():
            log.warning('Invalid score.scaled=%s found in SCORM state. attempt_id=%s', scaled_score, attempt.id)
            return
        if attempt.scaled_score != scaled_score:
            attempt.scaled_score = scaled_score
            attempt._scaled_score_updated = True


def update_scorm_progress(attempt: ScormResourceStudentAttempt):
    if getattr(attempt, '_scaled_score_updated', False) and attempt.scaled_score.is_finite():
        score = int((attempt.scaled_score * Decimal(100)).quantize(
            Decimal('1.'), rounding=ROUND_HALF_UP
        ))
        attempt.scorm_resource.scorm_file.scorm.update_progress(student=attempt.student, value=score)
