from djangorestframework_camel_case.parser import CamelCaseJSONParser
from djangorestframework_camel_case.render import CamelCaseJSONRenderer
from drf_spectacular.utils import extend_schema

from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError as DjangoValidationError
from django.db import IntegrityError
from django.http import Http404
from django.shortcuts import get_object_or_404
from django.utils.translation import gettext
from django.utils.translation import gettext_lazy as _

from rest_framework import serializers, status
from rest_framework.decorators import action
from rest_framework.exceptions import NotFound, ValidationError
from rest_framework.response import Response

from lms.core.views.viewsets import APIModelViewSet
from lms.courses.models import CourseStudent
from lms.courses.permissions import IsCourseStudent
from lms.courses.views.api import CourseModuleBaseViewSet

from ..models import Scorm, ScormResourceStudentAttempt, ScormStudentAttempt
from ..serializers import (
    ScormDetailSerializer, ScormResourceStudentAttemptDetailSerializer, ScormResourceStudentAttemptUpdateSerializer,
    ScormStudentAttemptCreateSerializer, ScormStudentAttemptDetailSerializer,
)
from ..services import create_new_attempt

User = get_user_model()


class ScormViewSet(CourseModuleBaseViewSet):
    """
    SCORM-модуль
    """
    serializer_class = ScormDetailSerializer
    queryset = Scorm.objects.active().prefetch_related('current_file')
    pagination_class = None

    @extend_schema(
        summary=gettext("Информация о scorm-модуле"),
        request=ScormDetailSerializer,
        responses={200: ScormDetailSerializer},
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)


class ScormStudentAttemptViewSet(APIModelViewSet):
    """
    Данные пользователя по SCORM-модулю
    """
    permission_classes = [IsCourseStudent, ]
    serializer_class = ScormStudentAttemptDetailSerializer
    serializer_classes = {
        'create': ScormStudentAttemptCreateSerializer,
    }
    queryset = ScormStudentAttempt.objects.filter(
        student__status=CourseStudent.StatusChoices.ACTIVE,
    ).select_related('scorm_file').prefetch_related('scorm_file__resources')
    pagination_class = None
    lookup_url_kwarg = 'pk'
    lookup_field = 'scorm_id'
    _cached_scorm = None

    def get_queryset(self):
        queryset = super().get_queryset()

        user = getattr(self.request, 'user', None)
        if user and isinstance(user, User):
            return queryset.filter(student__user=user)

        return queryset.none()

    def get_scorm(self) -> Scorm:
        if not self._cached_scorm:
            scorm_id = self.kwargs.get('pk')
            if self.action == 'create':
                scorm_id = self.request.data.get('scorm_id')
            queryset = Scorm.objects.select_related('course')
            self._cached_scorm = get_object_or_404(queryset, pk=scorm_id)
        return self._cached_scorm

    def get_course(self):
        return self.get_scorm().course

    @extend_schema(
        summary=gettext("Данные студента по scorm-модулю"),
        request=ScormStudentAttemptDetailSerializer,
        responses={200: ScormStudentAttemptDetailSerializer},
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Новая попытка студента по scorm-модулю"),
        request=None,
        responses={200: ScormStudentAttemptDetailSerializer},
    )
    @action(detail=True)
    def new_attempt(self, request, pk):
        user = getattr(self.request, 'user', None)
        if user and isinstance(user, User):
            scorm = get_object_or_404(Scorm, id=pk)
            student = get_object_or_404(
                CourseStudent,
                user=user, course=scorm.course, status=CourseStudent.StatusChoices.ACTIVE,
            )
            try:
                attempt = create_new_attempt(scorm=scorm, student=student)
                response_serializer = self.get_retrieve_serializer(attempt)
                headers = self.get_success_headers(response_serializer.data)
                return Response(response_serializer.data, headers=headers)
            except DjangoValidationError as exc:
                raise ValidationError(detail=serializers.as_serializer_error(exc))
            except IntegrityError:
                raise ValidationError(_('Попытка уже создана'), code='unique')

    @extend_schema(
        summary=gettext("Новая попытка студента по scorm-модулю"),
        request=ScormStudentAttemptCreateSerializer,
        responses={201: ScormStudentAttemptDetailSerializer},
    )
    def create(self, request, *args, **kwargs):
        scorm = self.get_scorm()
        student = get_object_or_404(
            CourseStudent,
            user_id=self.request.user.id, course_id=scorm.course_id, status=CourseStudent.StatusChoices.ACTIVE,
        )
        instance = self.get_queryset().filter(scorm_id=scorm.id, student_id=student.id).first()
        serializer = self.get_serializer(instance=instance, data=request.data)
        serializer.is_valid(raise_exception=True)

        try:
            serializer.save(scorm=scorm, student=student, scorm_file_id=scorm.current_file_id)
        except DjangoValidationError as exc:
            raise ValidationError(detail=serializers.as_serializer_error(exc))
        except IntegrityError:
            raise ValidationError(_('Попытка уже создана'), code='unique')

        response_serializer = self.get_retrieve_serializer(serializer.instance)
        headers = self.get_success_headers(response_serializer.data)
        return Response(response_serializer.data, status=status.HTTP_201_CREATED, headers=headers)


class CamelCaseWithoutDataJSONRenderer(CamelCaseJSONRenderer):
    json_underscoreize = {'ignore_fields': ('data',)}


class CamelCaseWithoutDataJSONParser(CamelCaseJSONParser):
    json_underscoreize = {'ignore_fields': ('data',)}


class ScormResourceStudentAttemptViewSet(APIModelViewSet):
    """
    Данные пользователя по SCORM-ресурсу
    """
    permission_classes = [IsCourseStudent, ]
    serializer_class = ScormResourceStudentAttemptDetailSerializer
    serializer_classes = {
        'update': ScormResourceStudentAttemptUpdateSerializer,
    }
    queryset = ScormResourceStudentAttempt.objects.all()
    pagination_class = None
    parser_classes = (CamelCaseWithoutDataJSONParser,)
    renderer_classes = (CamelCaseWithoutDataJSONRenderer,)
    lookup_url_kwarg = 'pk'
    lookup_field = 'scorm_resource_id'
    _cached_attempt = None

    def get_queryset(self):
        queryset = super().get_queryset()
        user = getattr(self.request, 'user', None)

        if not user:
            return queryset.none()

        return queryset.filter(
            student__user=user, student__status=CourseStudent.StatusChoices.ACTIVE,
            current_attempt=self.get_attempt().current_attempt,
        )

    def get_attempt(self):
        if not self._cached_attempt:
            try:
                self._cached_attempt = get_object_or_404(
                    ScormStudentAttempt.objects.select_related('scorm__course'),
                    scorm_file__resources=self.kwargs['pk'],
                    student__user=self.request.user,
                    student__status=CourseStudent.StatusChoices.ACTIVE,
                )
            except Http404:
                raise NotFound(_("Не найдена текущая попытка прохождения SCORM-курса"))
        return self._cached_attempt

    def get_course(self):
        return self.get_attempt().scorm.course

    @extend_schema(
        summary=gettext("Данные студента по scorm-ресурсу"),
        request=ScormResourceStudentAttemptDetailSerializer,
        responses={200: ScormResourceStudentAttemptDetailSerializer},
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Обновить данные студента по scorm-ресурсу"),
        request=ScormResourceStudentAttemptUpdateSerializer,
        responses={200: ScormResourceStudentAttemptDetailSerializer},
    )
    def update(self, request, *args, **kwargs):
        return super().update(request, *args, **kwargs)
