from drf_spectacular.utils import extend_schema

from django.core.exceptions import ValidationError as DjangoValidationError
from django.shortcuts import get_object_or_404
from django.utils.translation import gettext

from rest_framework import serializers
from rest_framework.exceptions import ValidationError

from lms.core.views.pagination import LimitOffsetAllPagination
from lms.core.views.viewsets import LabModelViewSet
from lms.courses.permissions import CourseObjectPermission
from lms.courses.views import GetCourseMixin

from ..models import Scorm, ScormFile
from ..serializers import (
    ScormCreateLabSerializer, ScormDetailLabSerializer, ScormFileCreateLabSerializer, ScormFileDetailLabSerializer,
    ScormFileListLabSerializer, ScormUpdateLabSerializer,
)


class ScormLabViewSet(GetCourseMixin, LabModelViewSet):
    serializer_class = ScormDetailLabSerializer
    serializer_classes = {
        'create': ScormCreateLabSerializer,
        'update': ScormUpdateLabSerializer,
        'partial_update': ScormUpdateLabSerializer,
    }
    queryset = Scorm.objects.select_related('course', 'current_file__course_file')\
        .prefetch_related('current_file__resources')
    pagination_class = LimitOffsetAllPagination
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            course_id = self.kwargs['pk']
            filter_kwargs = {'course_id': course_id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset

    @extend_schema(
        responses={200: ScormDetailLabSerializer},
        summary=gettext("Информация о scorm-модуле")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        request=ScormCreateLabSerializer,
        responses={201: ScormDetailLabSerializer},
        summary=gettext("Создание scorm-модуля"),
    )
    def create(self, request, *args, **kwargs):
        return super().create(request, *args, **kwargs)

    @extend_schema(
        request=ScormUpdateLabSerializer,
        responses={200: ScormDetailLabSerializer},
        summary=gettext("Обновление scorm-модуля"),
    )
    def update(self, request, *args, **kwargs):
        return super().update(request, *args, **kwargs)

    @extend_schema(
        request=ScormUpdateLabSerializer,
        responses={200: ScormDetailLabSerializer},
        summary=gettext("Частичное обновление scorm-модуля"),
    )
    def partial_update(self, request, *args, **kwargs):
        return super().partial_update(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление scorm-модуля"),
    )
    def destroy(self, request, *args, **kwargs):
        return super().destroy(request, *args, **kwargs)

    def perform_destroy(self, instance: Scorm):
        try:
            return instance.delete()
        except DjangoValidationError as exc:
            raise ValidationError(detail=serializers.as_serializer_error(exc))


class ScormFileLabViewSet(GetCourseMixin, LabModelViewSet):
    serializer_class = ScormFileListLabSerializer
    serializer_classes = {
        'create': ScormFileCreateLabSerializer,
        'retrieve': ScormFileDetailLabSerializer
    }
    queryset = ScormFile.objects.select_related(
        'scorm', 'course_file'
    ).prefetch_related(
        'resources'
    ).order_by('id')
    pagination_class = LimitOffsetAllPagination
    permission_classes = LabModelViewSet.permission_classes + [CourseObjectPermission]

    _cached_scorm = None

    def get_scorm(self, pk) -> 'Scorm':
        if not self._cached_scorm:
            queryset = Scorm.objects.select_related('course')
            self._cached_scorm = get_object_or_404(queryset, pk=pk)
        return self._cached_scorm

    def get_course(self, obj=None):
        if obj:
            return obj.scorm.course

        if self.action == 'create':
            return self.get_scorm(self.request.data.get('scorm_id')).course
        if self.action == 'list':
            return self.get_scorm(self.kwargs['pk']).course

    def get_queryset(self):
        queryset = super().get_queryset()
        if self.action == 'list':
            scorm_id = self.kwargs['pk']
            filter_kwargs = {'scorm_id': scorm_id}
            queryset = queryset.filter(**filter_kwargs)

        return queryset

    @extend_schema(
        request=ScormFileCreateLabSerializer,
        responses={201: ScormFileDetailLabSerializer},
        summary=gettext("Добавление новой версии файла"),
    )
    def create(self, request, *args, **kwargs):
        return super().create(request, *args, **kwargs)

    @extend_schema(
        responses={200: ScormFileDetailLabSerializer},
        summary=gettext("Информация о версии файла")
    )
    def retrieve(self, request, *args, **kwargs):
        return super().retrieve(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Список всех версий файлов")
    )
    def list(self, request, *args, **kwargs):
        return super().list(request, *args, **kwargs)

    @extend_schema(
        summary=gettext("Удаление scorm-модуля"),
    )
    def destroy(self, request, *args, **kwargs):
        return super().destroy(request, *args, **kwargs)
