import logging

from django_object_actions import DjangoObjectActions, takes_instance_or_queryset

from django.contrib import admin, messages
from django.db.models.query import QuerySet
from django.utils.translation import gettext_lazy as _

from .models import (
    StaffCity, StaffCountry, StaffGroup, StaffGroupForSupport, StaffLeadership, StaffOffice, StaffOrganization,
    StaffProfile, StaffProfileForSupport,
)

logger = logging.getLogger(__name__)


class BaseStaffGroupAdmin(DjangoObjectActions, admin.ModelAdmin):
    change_actions = [
        'update_group',
    ]

    changelist_actions = [
        'sync_staff_groups',
    ]

    def update_group(self, request, obj: StaffGroup):
        from .services import load_staff_groups

        try:
            load_staff_groups(group_ids=obj.pk)
            return self.message_user(request, _("Данные по подразделению были обновлены"))
        except Exception:
            return self.message_user(request, _("Ошибка при обновлении данных со Стаффа"), messages.ERROR)

    update_group.label = _("обновить со стаффа")

    def sync_staff_groups(self, request, queryset):
        from .tasks import update_staff_groups

        try:
            update_staff_groups.delay()
            return self.message_user(request, _("Синхронизация подразделений со Стаффа запущена"))
        except Exception:
            logger.exception("Error while sync staff departments")
            return self.message_user(request, _("Ошибка при запуске синхронизации данных со Стаффа"), messages.ERROR)

    sync_staff_groups.label = _("синхр. подразделений со Стаффа")


@admin.register(StaffGroup)
class StaffGroupAdmin(BaseStaffGroupAdmin):
    list_display = ('name', 'level', 'is_active', 'created', 'modified')
    search_fields = ('name',)
    fields = (
        'name', 'level', 'group_type', 'joined_at', 'is_active',
    )


@admin.register(StaffGroupForSupport)
class StaffGroupForSupportAdmin(BaseStaffGroupAdmin):
    list_display = ('name',)
    search_fields = ('name',)
    fields = (
        'name',
    )


@admin.register(StaffOffice)
class StaffOfficeAdmin(admin.ModelAdmin):
    list_display = ('name', 'is_active', 'created', 'modified')


@admin.register(StaffOrganization)
class StaffOrganizationAdmin(admin.ModelAdmin):
    list_display = ('name', 'is_active')


@admin.register(StaffCity)
class StaffCityAdmin(admin.ModelAdmin):
    list_display = ('name', 'is_active', 'created', 'modified')


@admin.register(StaffCountry)
class StaffCountryAdmin(admin.ModelAdmin):
    list_display = ('name', 'is_active', 'created', 'modified')


class StaffLeadershipInline(admin.TabularInline):
    model = StaffLeadership
    fields = ('group', 'role', 'joined_at')
    extra = 0


class StaffGroupInline(admin.TabularInline):
    model = StaffProfile.groups.through
    fields = ('staffgroup', 'display_group_level', 'display_group_type')
    readonly_fields = ('staffgroup', 'display_group_level', 'display_group_type')
    verbose_name_plural = _("подразделения")
    verbose_name = _("подразделение")
    extra = 0

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.select_related('staffgroup')

    def display_group_level(self, obj: StaffProfile.groups):
        return obj.staffgroup.level

    display_group_level.short_description = _("уровень")

    def display_group_type(self, obj: StaffProfile.groups):
        return obj.staffgroup.get_group_type_display()

    display_group_type.short_description = _("тип")


class BaseStaffProfileAdmin(DjangoObjectActions, admin.ModelAdmin):
    actions = [
        'update_profile',
    ]

    change_actions = [
        'update_profile',
    ]

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.select_related('user', 'city', 'office')

    @takes_instance_or_queryset
    def update_profile(self, request, queryset: QuerySet):
        from .services import load_staff_profile

        ok_ids = []
        error_ids = []
        for obj in queryset:
            if load_staff_profile(obj.user_id, force_load=True):
                ok_ids.append(obj.id)
            else:
                error_ids.append(obj.id)

        if error_ids:
            return self.message_user(
                request, _(f"Ошибка при обновлении данных профилей {error_ids} со Стаффа"), messages.ERROR,
            )

        return self.message_user(request, _(f"Данные профилей {ok_ids} были обновлены"))

    update_profile.short_description = _("обновить со стаффа")
    update_profile.label = _("обновить со стаффа")

    def has_change_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False


@admin.register(StaffProfile)
class StaffProfileAdmin(BaseStaffProfileAdmin):
    inlines = [
        StaffGroupInline,
        StaffLeadershipInline,
    ]
    list_display = ('user', 'joined_at', 'created', 'modified')
    list_select_related = ('user', 'city', 'office', 'organization',)
    fields = (
        'user',
        ('first_name_ru', 'last_name_ru'),
        ('first_name_en', 'last_name_en'),
        'office', 'city', 'organization', 'position_ru', 'position_en', 'language_native',
        'joined_at', 'dismissed_at', 'is_active', 'is_homeworker', 'is_dismissed',
        'head', 'hr_partners', 'groups_tree',
    )
    search_fields = ('user__username', 'first_name_ru', 'last_name_ru', 'first_name_en', 'last_name_en')
    filter_horizontal = (
        'hr_partners',
    )


@admin.register(StaffProfileForSupport)
class StaffProfileForSupportAdmin(BaseStaffProfileAdmin):
    list_display = ('user',)
    list_select_related = ('user',)
    fields = (
        'user',
    )
    search_fields = ('user__username',)
