import logging
import time
from collections import Counter
from datetime import datetime
from functools import wraps
from typing import Dict, Iterable, List, Optional, Union

from django.contrib.auth import get_user_model
from django.db import connection, reset_queries, transaction
from django.db.models.signals import post_save
from django.db.utils import DataError
from django.utils import timezone

from lms.contrib.staff.client import staff_api

from .models import StaffCity, StaffCountry, StaffGroup, StaffLeadership, StaffOffice, StaffOrganization, StaffProfile
from .utils import lookup_field, parse_fields

User = get_user_model()

log = logging.getLogger(__name__)


STAFF_API_DEFAULT_LIMIT = 1000


def modified_user_post_save_handler(instance: User, created, **kwargs):
    if created or not getattr(instance, 'staffprofile', None):
        StaffProfile.objects.get_or_create(user=instance)


class CacheManager:
    """
    Менеджер кэшей в памяти
    """
    def __init__(self):
        self._caches = {}

    def set(self, key, value):
        self._caches[key] = value

    def get(self, key, cast=dict):
        if not self._caches.get(key):
            self._caches[key] = cast()
        return self._caches[key]

    def clear(self, key) -> None:
        cache = self._caches.get(key)
        if cache:
            cache.clear()

    def clear_all(self) -> None:
        self._caches.clear()


class Counts(Counter):
    """
    Счетчики с информативным выводом
    """
    def __str__(self):
        return ', '.join([f'{k}: {v}' for k, v in self.items()])


def cached_id_into_set(name, key='id'):
    """
    Декоратор методов класса, для кеширования id-шников в set'ах
    """
    def wrapper(method):
        @wraps(method)
        def decorator(self, data, *args, **kwargs):
            cache = self.cache.get(name, cast=set)
            data_key = data.get(key)

            # кеширование можно отключать,
            # если у класса выставить параметр: no_cached_<cache_name> = True
            cache_disabled = getattr(self, f'no_cached_{name}', False)
            if cache_disabled:
                return method(self, data, *args, **kwargs)

            if data_key is None:
                raise ValueError(f'Key {key} not found')

            if data_key not in cache:
                try:
                    return method(self, data, *args, **kwargs)
                finally:
                    cache.add(data_key)
                    # считаем кол-во элементов к кэше
                    self.counts[f'{name}_size'] += 1
            else:
                # считаем кол-во попаданий в кэш
                self.counts[f'{name}_hits'] += 1
                return data_key

        return decorator

    return wrapper


class BaseLoader:
    """
    Базовый загрузчик со Стаффа
    """
    endpoint = None
    init_kwargs = None
    fields = None
    limit = STAFF_API_DEFAULT_LIMIT
    max_limit = None

    def __init__(self, limit=None, max_limit=None, **kwargs):
        if limit:
            self.limit = limit

        self.max_limit = max_limit
        self.cache = CacheManager()
        self.counts = Counts()

    def clear_cache(self):
        self.cache.clear_all()

    def get_init_kwargs(self) -> dict:
        return dict(self.init_kwargs or {})

    def get_fields(self):
        if self.fields:
            return parse_fields(self.fields)

    def request(self, **kwargs):
        params = self.get_init_kwargs() or {}
        params['_limit'] = self.limit

        fields = self.get_fields()
        if fields:
            params['_fields'] = ','.join(set(fields))

        if kwargs:
            params.update(**kwargs)

        response = getattr(staff_api, self.endpoint).get(**params)

        return response

    def get_items(self, **kwargs):
        response = self.request(**kwargs)
        return self.process_data(response)

    def process_data(self, response, **kwargs):
        has_results = True

        while has_results:
            result = response.get('result', [])
            page = response.get('page', 1)
            pages = response.get('pages', 1)

            for item in result:
                yield item

            if self.max_limit:
                current_limit = self.limit * int(page)
                if current_limit >= self.max_limit:
                    break

            if page < pages:
                page += 1
                response = self.request(_page=page)
                continue

            has_results = False

        return False

    def update(self, **kwargs):
        return NotImplementedError('method update must be implemented')


class BaseProfileLoader(BaseLoader):
    """
    Базовый загрузчик профилей пользователей
    """
    uid_field = 'uid'

    def __init__(self, uids: Optional[Union[Iterable[int], int]] = None, **kwargs):
        if isinstance(uids, int):
            uids = [uids]
        self.uids = uids
        self.loaded_uids = set()

        super().__init__(**kwargs)

    def get_init_kwargs(self) -> dict:
        kwargs = super().get_init_kwargs()
        if self.uids:
            kwargs[self.uid_field] = ','.join(map(str, self.uids))
        else:
            kwargs.setdefault('is_deleted', False)
            kwargs.setdefault('official.is_dismissed', False)
            kwargs.setdefault('official.is_robot', False)
        return kwargs

    def get_profiles(
        self, uids: Iterable[int],
        force_update: bool = False
    ) -> Dict[int, 'StaffProfile']:
        cached_profiles = self.cache.get('profiles')
        profiles = {}
        uids_set = set(uids)

        if not force_update:
            for uid in uids_set.intersection(cached_profiles):
                profiles[uid] = cached_profiles[uid]
                uids_set.remove(uid)
            self.counts['profile_hits'] += len(profiles)

        filters = {
            f"{StaffProfile.YAUID_FIELD}__in": uids_set,
        }
        select_fields = getattr(self, 'select_fields', [])
        prefetch_fields = getattr(self, 'prefetch_fields', [])
        queryset = StaffProfile.objects.select_related(*select_fields, 'user').filter(**filters)
        if prefetch_fields:
            queryset = queryset.prefetch_related(*prefetch_fields)
        for profile in queryset:
            uid = profile.uid
            cached_profiles[uid] = profile
            profiles[uid] = profile

        return profiles

    def profile_data(self, batch_len: int = 500, **kwargs):
        items = []
        uids = set()
        for item in self.get_items():
            uid = lookup_field(item, self.uid_field)
            if not uid:
                log.warning('No uid found: %r', item)
                continue

            try:
                uid = int(uid)
            except ValueError:
                log.warning(f'Wrong uid: {uid}')
                continue

            items.append((uid, item))
            uids.add(uid)

            if len(items) == batch_len:
                profiles = self.get_profiles(uids)
                for profile_uid, data in items:
                    yield profiles[profile_uid], data
                items.clear()
                uids.clear()

        if items:
            profiles = self.get_profiles(uids)
            for profile_uid, data in items:
                yield profiles[profile_uid], data

    def update(self, **kwargs):
        from lms.courses.services import flush_cache_course_available_for, flush_cache_courses_unavailable_for

        from .receivers import staff_profile_post_save_handler

        post_save.disconnect(receiver=staff_profile_post_save_handler, sender=StaffProfile)

        self.loaded_uids.clear()
        updated_users = set()

        try:
            for profile, data in self.profile_data():
                with transaction.atomic():
                    self.update_profile(profile, data)
                    updated_users.add(profile.user_id)
                self.counts['processed'] += 1

        finally:
            post_save.connect(receiver=staff_profile_post_save_handler, sender=StaffProfile)
            if updated_users:
                flush_cache_courses_unavailable_for(users_ids=updated_users)
                flush_cache_course_available_for(users_ids=updated_users)

        log.debug('Summary: %s', self.counts)
        self.clear_cache()

    def update_profile(self, profile: 'StaffProfile', data: dict):
        return NotImplementedError('method update_profile must be implemented')


class ModelMixin:
    def fill_model_cache(self, model):
        model_name = model.__name__.lower()
        self.cache.set(model_name, set(model.objects.values_list('pk', flat=True)))

    def method_or_create(self, model, method, data: dict, **kwargs):
        model_name = model.__name__.lower()
        pk = data['id']
        created = False

        cache = self.cache.get(model_name, cast=set)
        no_cache = kwargs.pop('no_cache', False)

        if pk not in cache or no_cache:
            extracted = model.extract_from_dict(data)
            if kwargs:
                extracted.update(**kwargs)

            _, created = method(id=pk, defaults=extracted)
            cache.add(pk)
            self.counts[f'{model_name}_size'] += 1
        else:
            if not no_cache:
                self.counts[f'{model_name}_hits'] += 1

        if created:
            self.counts[f'{model_name}_created'] += 1

        return pk

    def get_or_create_model(self, model, data: dict, **kwargs):
        return self.method_or_create(
            model,
            model.objects.get_or_create,
            data,
            **kwargs,
        )

    def update_or_create_model(self, model, data: dict, **kwargs):
        return self.method_or_create(
            model,
            model.objects.update_or_create,
            data,
            **kwargs,
        )


class GroupMixin:
    """
    Миксин для обновления подразделений со Стаффа
    """
    def update_or_create_group(self, group: dict, **kwargs):
        return self.update_or_create_model(StaffGroup, group, **kwargs)

    def get_or_create_group(self, group: dict, **kwargs):
        return self.get_or_create_model(StaffGroup, group, **kwargs)


class UserMixin:
    """
    Миксин для создания базовых профилей пользователей
    """
    def get_user_cache_key(self, user_data: dict) -> str:
        return f"{user_data['uid']}:{user_data['login']}"

    def prepare_user_data(self, user_data: dict) -> dict:
        defaults = {
            User.YAUID_FIELD: user_data['uid'],
        }

        first_name = lookup_field(user_data, 'name.first.ru') or lookup_field(user_data, 'name.first.en')
        if first_name:
            defaults['first_name'] = first_name

        last_name = lookup_field(user_data, 'name.last.ru') or lookup_field(user_data, 'name.last.en')
        if last_name:
            defaults['last_name'] = last_name

        return defaults

    def create_user(self, user_data: dict):
        cache_key = self.get_user_cache_key(user_data)
        cached_users = self.cache.get('users')

        if not cached_users.get(cache_key):
            user_fields = self.prepare_user_data(user_data)
            user_fields[User.USERNAME_FIELD] = user_data['login']
            obj = User.objects.create(**user_fields)
            cached_users[cache_key] = obj
            self.counts['user_created'] += 1
            self.counts['users_size'] += 1
        else:
            self.counts['users_hits'] += 1

        return cached_users[cache_key]

    def get_or_create_user(self, user_data: dict):
        cache_key = self.get_user_cache_key(user_data)
        cached_users = self.cache.get('users')

        if not cached_users.get(cache_key):
            user, user_created = User.objects.update_or_create(
                **{User.USERNAME_FIELD: user_data['login']},
                defaults=self.prepare_user_data(user_data),
            )
            if user_created:
                self.counts['user_created'] += 1

            cached_users[cache_key] = user
            self.counts['users_size'] += 1

        else:
            self.counts['users_hits'] += 1

        return cached_users[cache_key]


class StaffUserLoader(UserMixin, BaseLoader):
    """
    Загрузчик данных профиля со Стаффа и создания пользователя
    """
    endpoint = 'persons'
    login_field = 'login'
    fields = [
        '_meta',
        'uid',
        'name',
        'login',
        'is_deleted',
    ]

    def __init__(self, logins: Iterable[str], **kwargs):
        self.logins = logins
        super().__init__(**kwargs)

    def get_init_kwargs(self) -> dict:
        kwargs = super().get_init_kwargs()
        kwargs[self.login_field] = ','.join(map(str, self.logins))
        return kwargs

    def get_or_create(self) -> List[User]:
        found_users = []
        for data in self.get_items():
            with transaction.atomic():
                found_users.append(self.get_or_create_user(data))
            self.counts['processed'] += 1

        log.debug('Summary: %s', self.counts)
        self.clear_cache()

        return found_users


class StaffOfficeLoader(ModelMixin, BaseLoader):
    """
    Загрузчик всех офисов (а также городов и стран)
    """
    endpoint = 'offices'
    fields = [
        'id',
        'name',
        'code',
        'is_deleted',
        {'city': ('id', 'name', 'is_deleted')},
        {'city.country': ('id', 'name', 'code', 'is_deleted')},
    ]

    def update(self, **kwargs) -> None:
        self.counts['created'] = 0

        for office in self.get_items():
            city = office.get('city', {})
            country = city.get('country', {})

            self.update_or_create_model(StaffCountry, country)
            self.update_or_create_model(StaffCity, city, country_id=country['id'])
            self.update_or_create_model(StaffOffice, office, city_id=city['id'], no_cache=True)

            self.counts['processed'] += 1

        log.debug('Summary: %s', self.counts)
        self.clear_cache()


class StaffGroupLoader(GroupMixin, ModelMixin, BaseLoader):
    """
    Загрузчик подразделений со Стаффа (включая родительские)
    """
    endpoint = 'groups'
    fields = [
        'id', 'name', 'type', 'level', 'is_deleted',
    ]
    init_kwargs = {
        'type': 'department',
    }

    def __init__(self, group_ids: Optional[Union[list, int]] = None, **kwargs):
        if group_ids and isinstance(group_ids, int):
            group_ids = [group_ids]
        self.group_ids = group_ids

        super().__init__(**kwargs)

    def get_init_kwargs(self) -> dict:
        kwargs = super().get_init_kwargs()
        if self.group_ids:
            kwargs['id'] = ','.join(map(str, self.group_ids))
        return kwargs

    def update(self, **kwargs) -> None:
        self.counts['created'] = 0

        for group in self.get_items(**kwargs):
            self.update_or_create_group(group)
            self.counts['processed'] += 1

        log.debug('Summary: %s', self.counts)
        self.clear_cache()


class StaffProfileProfileLoader(UserMixin, ModelMixin, BaseProfileLoader):
    """
    Загрузчик данных профиля со Стаффа
    """
    endpoint = 'persons'
    uid_field = 'uid'
    fields = [
        '_meta',
        'uid',
        'name',
        'login',
        'is_deleted',
        {'hr_partners': ('uid', 'login')},
        {'chief': ('uid', 'login', 'name')},
        {'location.office': ('id', 'code', 'name', 'is_deleted', 'city')},
        {'official': ('join_at', 'quit_at', 'is_dismissed', 'is_homeworker', 'position')},
        {'official.organization': ('id', 'name', 'name_en', 'is_deleted')},
        'language.native',
    ]

    batch_len = 500

    def preload_cache(self):
        # загружаем текущие значения справочников в кэш
        self.fill_model_cache(StaffOrganization)
        self.fill_model_cache(StaffCountry)
        self.fill_model_cache(StaffCity)
        self.fill_model_cache(StaffOffice)

    def update_profile(self, profile: StaffProfile, data: dict):
        log.debug('%s: update profile: %s', __class__.__name__, profile)
        profile.is_active = not data.get('is_deleted', False)

        official = data.get('official')
        if official:
            join_at = official.get('join_at')
            if join_at:
                try:
                    join_at = datetime.strptime(join_at, '%Y-%m-%d').replace(tzinfo=timezone.utc)
                    profile.joined_at = join_at
                except ValueError:
                    pass

            quit_at = official.get('quit_at')
            if quit_at:
                try:
                    quit_at = datetime.strptime(quit_at, '%Y-%m-%d').replace(tzinfo=timezone.utc)
                    profile.dismissed_at = quit_at
                except ValueError:
                    pass

            profile.is_homeworker = official.get('is_homeworker', False)
            profile.is_dismissed = official.get('is_dismissed', False)
            profile.position_ru = lookup_field(official, 'position.ru') or ''
            profile.position_en = lookup_field(official, 'position.en') or ''

        organization = official.get('organization')
        if organization:
            organization_id = organization.get('id')
            if organization_id:
                if profile.organization_id != organization_id:
                    self.get_or_create_model(StaffOrganization, organization)
                    profile.organization_id = organization_id
            else:
                log.warning('Missing organization id: %s', str(data))

        chief = data.get('chief')
        if chief:
            chief_profile = self.get_or_create_user(user_data=chief).staffprofile
            profile.head_id = chief_profile.id

        office = lookup_field(data, 'location.office')
        if office and profile.office_id != office['id']:
            city = office.get('city', {})
            country = city.get('country', {})

            self.get_or_create_model(StaffCountry, country)
            self.get_or_create_model(StaffCity, city, country_id=country['id'])
            self.get_or_create_model(StaffOffice, office, city_id=city['id'])

            profile.office_id = office['id']

        if data.get('name'):
            profile.first_name_ru = lookup_field(data, 'name.first.ru')
            profile.last_name_ru = lookup_field(data, 'name.last.ru')
            profile.first_name_en = lookup_field(data, 'name.first.en')
            profile.last_name_en = lookup_field(data, 'name.last.en')

        language_native = lookup_field(data, 'language.native')
        if language_native:
            profile.language_native = language_native

        profile.save()

        user = profile.user
        if user.first_name != profile.first_name_ru or user.last_name != profile.last_name_ru:
            User.objects.filter(id=user.id).update(first_name=profile.first_name_ru, last_name=profile.last_name_ru)

    def _batch_update(self, profiles: dict):
        filters = {
            f"{StaffProfile.YAUID_FIELD}__in": profiles.keys(),
        }
        queryset = StaffProfile.objects.select_related('user').filter(**filters)
        existed = {str(p.uid): p for p in queryset}
        for uid, new_data in profiles.items():
            current_profile = existed.get(str(uid), None)
            with transaction.atomic():
                try:
                    if current_profile is None:
                        user = self.create_user(new_data)
                        current_profile = user.staffprofile
                    self.update_profile(current_profile, new_data)
                except DataError as e:
                    log.exception(f"Failed to update profile {uid} ({e}), data: {new_data}")

            self.counts['processed'] += 1
            self.loaded_uids.add(int(uid))

        self.counts['profile_queries'] += len(connection.queries)
        reset_queries()

    def update_or_create(self):
        start = time.time()

        from lms.mailing.receivers import staff_profile_in_mailing_post_save_handler

        from .receivers import staff_profile_post_save_handler, user_post_save_handler
        self.loaded_uids.clear()

        post_save.disconnect(receiver=user_post_save_handler, sender=User)
        post_save.disconnect(receiver=staff_profile_post_save_handler, sender=StaffProfile)
        post_save.disconnect(receiver=staff_profile_in_mailing_post_save_handler, sender=StaffProfile)
        post_save.connect(receiver=modified_user_post_save_handler, sender=User)

        log.info('Preload cache...')
        self.preload_cache()
        log.info('Preload complete.')

        self.counts['queries_before'] = len(connection.queries)
        reset_queries()

        try:
            profiles = {}
            for data in self.get_items():
                uid = lookup_field(data, self.uid_field)
                if not uid:
                    log.warning('No uid found: %r', data)
                    continue

                profiles[uid] = data

                if len(profiles) == self.batch_len:
                    self._batch_update(profiles)
                    profiles = {}

            if len(profiles):
                self._batch_update(profiles)

            log.debug('Summary: %s', self.counts)
            self.clear_cache()

        finally:
            post_save.disconnect(receiver=modified_user_post_save_handler, sender=User)
            post_save.connect(receiver=staff_profile_in_mailing_post_save_handler, sender=StaffProfile)
            post_save.connect(receiver=staff_profile_post_save_handler, sender=StaffProfile)
            post_save.connect(receiver=user_post_save_handler, sender=User)

        log.debug('total time: %i', time.time() - start)


class StaffGroupProfileLoader(GroupMixin, ModelMixin, BaseProfileLoader):
    """
    Загрузчик данных о группах пользователя со Стаффа
    """
    endpoint = 'groupmembership'
    uid_field = 'person.uid'
    init_kwargs = {
        'group.type': 'department',
    }
    fields = [
        {'person': ('uid',)},
        {
            'group': ('id', 'name', 'type', 'level', 'is_deleted',),
            'group.ancestors': ('id',),
        },
    ]
    prefetch_fields = ['groups']

    def update(self, **kwargs):
        from lms.courses.services import flush_cache_course_available_for, flush_cache_courses_unavailable_for
        from lms.mailing.receivers import staff_profile_in_mailing_post_save_handler

        from .receivers import staff_profile_post_save_handler

        updated_users = set()
        post_save.disconnect(receiver=staff_profile_post_save_handler, sender=StaffProfile)
        post_save.disconnect(receiver=staff_profile_in_mailing_post_save_handler, sender=StaffProfile)
        try:
            for profile, userdata in self.profile_data():
                group = userdata.get('group')
                if not group:
                    continue

                group['level'] = group.get('level', 0)
                self.get_or_create_group(group)

                ancestors = group.get('ancestors', [])
                groups_tree = [ancestor['id'] for ancestor in ancestors]
                groups_tree.append(group['id'])

                profile.groups.set([group['id']])

                if profile.groups_tree != groups_tree:
                    profile.groups_tree = groups_tree
                    profile.save()
                    updated_users.add(profile.user_id)

                self.counts['profiles_processed'] += 1
        finally:
            post_save.connect(receiver=staff_profile_post_save_handler, sender=StaffProfile)
            post_save.connect(receiver=staff_profile_in_mailing_post_save_handler, sender=StaffProfile)
            if updated_users:
                flush_cache_courses_unavailable_for(users_ids=updated_users)
                flush_cache_course_available_for(users_ids=updated_users)

        log.debug('Summary: %s', self.counts)
        self.clear_cache()


class StaffHrPartnersProfileLoader(UserMixin, BaseProfileLoader):
    """
    Загрузчик hr-партнеров
    """
    endpoint = 'groupmembership'
    uid_field = 'person.uid'
    init_kwargs = {
        'group.type': 'department',
    }
    fields = [
        {'person': ('uid', 'login', 'name', 'is_deleted')},
        {
            'group': ('id', 'type', 'name', 'level', 'is_deleted'),
            'group.department.heads': ('role', 'person.uid', 'person.login'),
            'group.ancestors': ('id', 'type', 'name', 'level', 'is_deleted'),
            'group.ancestors.department.heads': ('role', 'person.uid', 'person.login'),
        },
    ]
    prefetch_fields = ['hr_partners']

    def lookup_hr_partners(self, group):
        heads = lookup_field(group, 'department.heads', [])
        for head in heads:
            if head['role'] != 'hr_partner':
                continue

            yield head['person']

    def lookup_groups(self, userdata):
        """
        перебираем департаменты, начиная с текущего и дальше по родительским
        """
        group = userdata.get('group')
        if group:
            yield group

        for ancestor in reversed(group.get('ancestors', [])):
            if ancestor.get('is_deleted', False):
                continue

            yield ancestor

    def update(self, **kwargs):
        cached_profiles = self.cache.get('profiles', cast=set)

        for profile, userdata in self.profile_data():
            found = False

            if not getattr(profile, '_hr_profiles_pks', None):
                profile._hr_profiles_pks = set()

            for group in self.lookup_groups(userdata):
                for hr_partner in self.lookup_hr_partners(group):
                    hr_profile = self.get_or_create_user(hr_partner).staffprofile
                    profile._hr_profiles_pks.add(hr_profile.pk)
                    found = True

                if found:
                    break

            cached_profiles.add(profile)

        for profile in cached_profiles:
            if profile._hr_profiles_pks:
                profile.hr_partners.set(profile._hr_profiles_pks)
            else:
                profile.hr_partners.clear()
            self.counts['profile_processed'] += 1

        log.debug('Summary: %s', self.counts)
        self.clear_cache()


class StaffLeadershipProfileLoader(GroupMixin, ModelMixin, BaseProfileLoader):
    """
    Загрузчик данных о руководстве в группах на Стаффе
    """
    endpoint = 'departmentstaff'
    uid_field = 'person.uid'
    init_kwargs = {
        'department_group.type': 'department',
    }
    fields = [
        {'person': ('uid', 'login', 'is_deleted')},
        {'department_group': ('id', 'name', 'level', 'type', 'is_deleted')},
        'role',
    ]

    def update(self, **kwargs):
        cached_profiles = self.cache.get('profiles', cast=set)

        for profile, userdata in self.profile_data():
            group = userdata.get('department_group', {})
            if not group:
                continue

            staff_leadership = self.get_or_create_leadership(
                profile=profile,
                group=group['id'],
                role=userdata.get('role')
            )

            if not getattr(profile, '_staff_leadership_pks', None):
                profile._staff_leadership_pks = set()

            profile._staff_leadership_pks.add(staff_leadership.pk)
            cached_profiles.add(profile)

        # деактивируем членство в остальных группах
        for profile in cached_profiles:
            StaffLeadership.objects\
                .filter(profile=profile)\
                .exclude(pk__in=profile._staff_leadership_pks)\
                .update(is_active=False)

            self.counts['profiles_processed'] += 1

        log.debug('Summary: %s', self.counts)
        self.clear_cache()

    def get_or_create_leadership(self, **kwargs) -> 'StaffLeadership':
        staff_leadership, created = StaffLeadership.objects.get_or_create(**kwargs)

        if not created and not staff_leadership.is_active:
            staff_leadership.is_active = True
            staff_leadership.save()

        if created:
            self.counts['leaderships_created'] += 1

        return staff_leadership


class StaffLeadershipJoinedProfileLoader(BaseProfileLoader):
    """
    Загрузчик данных о начале руководстве пользователя со Стаффа
    """
    endpoint = 'groupmembership'
    uid_field = 'person.uid'
    init_kwargs = {
        'group.is_deleted': 'false',
        'group.type': 'wiki',
        'group.url': 'autoheads',
    }
    fields = [
        {'person': ('login', 'uid')},
        {'group': ('id', 'type', 'name', 'url')},
        'joined_at',
    ]

    def update_profile(self, profile: StaffProfile, data: dict):
        joined_at = data.get('joined_at')
        if not joined_at:
            return

        StaffLeadership.objects.filter(profile=profile).update(joined_at=joined_at)


class StaffDismissedProfileLoader(StaffProfileProfileLoader):
    """
    Загрузчик данных уволенных сотрудников

    (частичное обновление профиля пользователя)
    """
    endpoint = 'persons'
    fields = [
        '_meta',
        'uid',
        'is_deleted',
        'official.is_dismissed',
    ]
