from typing import List

from cache_memoize import cache_memoize
from model_utils import FieldTracker
from model_utils.models import TimeStampedModel

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.postgres.fields import JSONField
from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import gettext_lazy as _

from lms.contrib.staff.settings import STAFF_PROFILE_GROUP_NAMES_TIMEOUT
from lms.core.models.mixins import ActiveModelMixin

from .utils import get_language, lookup_field

User = get_user_model()


class TranslatedFieldMixin:
    def get_translated_field(self, field, default=None):
        lang = get_language()
        return getattr(self, f'{field}_{lang}', default)


class StaffCountry(TranslatedFieldMixin, TimeStampedModel, ActiveModelMixin):
    code = models.CharField(_("код"), max_length=20)
    name_ru = models.CharField(_("название RU"), max_length=255)
    name_en = models.CharField(_("название EN"), max_length=255, blank=True)

    class Meta:
        verbose_name = _("страна")
        verbose_name_plural = _("страны")

    @property
    def name(self) -> str:
        return self.get_translated_field('name', self.name_ru)

    def __str__(self):
        return self.name

    @staticmethod
    def extract_from_dict(data: dict) -> dict:
        return {
            'code': data.get('code'),
            'name_ru': lookup_field(data, 'name.ru'),
            'name_en': lookup_field(data, 'name.en'),
            'is_active': not data.get('is_deleted', False),
        }


class StaffCity(TranslatedFieldMixin, TimeStampedModel, ActiveModelMixin):
    country = models.ForeignKey(
        StaffCountry,
        verbose_name=_("страна"),
        related_name='cities',
        blank=True,
        null=True,
        on_delete=models.SET_NULL,
    )
    name_ru = models.CharField(_("название RU"), max_length=255)
    name_en = models.CharField(_("название EN"), max_length=255, blank=True)

    class Meta:
        verbose_name = _("город на Стаффе")
        verbose_name_plural = _("города")

    @property
    def name(self) -> str:
        return self.get_translated_field('name', self.name_ru)

    def __str__(self):
        return self.name

    @staticmethod
    def extract_from_dict(data: dict) -> dict:
        return {
            'name_ru': lookup_field(data, 'name.ru'),
            'name_en': lookup_field(data, 'name.en'),
            'is_active': not data.get('is_deleted', False),
        }


class StaffOffice(TranslatedFieldMixin, TimeStampedModel, ActiveModelMixin):
    city = models.ForeignKey(
        StaffCity,
        verbose_name=_("город"),
        related_name='offices',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    code = models.CharField(_("код"), max_length=255)
    name_ru = models.CharField(_("название RU"), max_length=255)
    name_en = models.CharField(_("название EN"), max_length=255, blank=True)

    class Meta:
        verbose_name = _("офис на Стаффе")
        verbose_name_plural = _("офисы")

    @property
    def name(self) -> str:
        return self.get_translated_field('name', self.name_ru)

    def __str__(self):
        return self.name

    @staticmethod
    def extract_from_dict(data: dict) -> dict:
        return {
            'name_ru': lookup_field(data, 'name.ru'),
            'name_en': lookup_field(data, 'name.en'),
            'is_active': not data.get('is_deleted', False),
        }


class StaffGroup(TimeStampedModel, ActiveModelMixin):
    level = models.PositiveIntegerField(_("уровень"), default=0)
    name = models.CharField(_("название"), max_length=255)

    TYPE_DEPARTMENT = 'department'
    TYPE_SERVICE = 'service'
    TYPE_SERVICEROLE = 'servicerole'
    TYPE_WIKI = 'wiki'

    _TYPE_CHOICES = (
        (TYPE_DEPARTMENT, _("department")),
        (TYPE_SERVICE, _("service")),
        (TYPE_SERVICEROLE, _("servicerole")),
        (TYPE_WIKI, _("wiki")),
    )
    group_type = models.CharField(_("тип"), max_length=20, choices=_TYPE_CHOICES, blank=True)
    joined_at = models.DateTimeField(_("дата добавления"), null=True, blank=True)

    class Meta:
        ordering = ('level', 'name',)
        verbose_name = _("подразделение на Стаффе")
        verbose_name_plural = _("подразделения")

    def __str__(self):
        return self.name

    @staticmethod
    def extract_from_dict(data: dict) -> dict:
        return {
            'name': data.get('name', ''),
            'level': data.get('level', ''),
            'group_type': data.get('type', ''),
            'is_active': not data.get('is_deleted', False),
        }


class StaffGroupForSupport(StaffGroup):
    class Meta:
        proxy = True
        ordering = ('level', 'name',)
        verbose_name = _("[Для поддержки] подразделение на Стаффе")
        verbose_name_plural = _("[Для поддержки] подразделения")


class StaffOrganization(TranslatedFieldMixin, TimeStampedModel, ActiveModelMixin):
    name_ru = models.CharField(_("название"), max_length=255)
    name_en = models.CharField(_("name"), max_length=255)

    class Meta:
        verbose_name = _("организация на Стаффе")
        verbose_name_plural = _("организации")

    def __str__(self):
        return self.name

    @property
    def name(self) -> str:
        return self.get_translated_field('name', self.name_ru)

    @staticmethod
    def extract_from_dict(data: dict) -> dict:
        return {
            'name_ru': lookup_field(data, 'name'),
            'name_en': lookup_field(data, 'name_en'),
            'is_active': not data.get('is_deleted', False),
        }


class StaffProfile(TranslatedFieldMixin, TimeStampedModel, ActiveModelMixin):
    user = models.OneToOneField(User, verbose_name=_("пользователь"), on_delete=models.CASCADE)
    joined_at = models.DateTimeField(_("дата выхода"), null=True, blank=True)
    dismissed_at = models.DateTimeField(_("дата увольнения"), null=True, blank=True)
    is_homeworker = models.BooleanField(_("надомник"), default=False)
    is_dismissed = models.BooleanField(_("бывший сотрудник"), default=False)
    language_native = models.CharField(_("родной язык"), max_length=2, blank=True, db_index=True)
    first_name_ru = models.CharField(_("имя (ru)"), max_length=30, blank=True)
    last_name_ru = models.CharField(_('фамилия (ru)'), max_length=150, blank=True)
    first_name_en = models.CharField(_('имя (en)'), max_length=30, blank=True)
    last_name_en = models.CharField(_('фамилия (en)'), max_length=150, blank=True)
    phones = JSONField(_("телефоны"), default=list, blank=True)
    position_ru = models.CharField(_("должность (ru)"), max_length=1024, blank=True)
    position_en = models.CharField(_("должность (en)"), max_length=1024, blank=True)
    head = models.ForeignKey(
        to='self',
        verbose_name=_("руководитель"),
        related_name='subordinates',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    hr_partners = models.ManyToManyField(
        to='self',
        verbose_name=_("hr-партнеры"),
        related_name='hr_partner_for',
        blank=True,
        symmetrical=False,
    )
    city = models.ForeignKey(
        StaffCity,
        verbose_name=_("город"),
        related_name='profiles',
        null=True,
        blank=True,
        on_delete=models.CASCADE,
    )
    office = models.ForeignKey(
        StaffOffice,
        verbose_name=_("офис"),
        related_name='profiles',
        null=True,
        blank=True,
        on_delete=models.CASCADE,
    )
    organization = models.ForeignKey(
        StaffOrganization,
        verbose_name=_("организация"),
        related_name='profiles',
        null=True,
        blank=True,
        on_delete=models.CASCADE,
    )

    groups = models.ManyToManyField(
        StaffGroup,
        blank=True,
        verbose_name=_("подразделения"),
    )

    groups_tree = JSONField(
        _("дерево подразделений"),
        default=list,
        blank=True,
    )

    YAUID_FIELD = 'user__{}'.format(User.YAUID_FIELD)

    tracker = FieldTracker(fields=['is_dismissed', 'office'])

    class Meta:
        verbose_name = _("профиль на Стаффе")
        verbose_name_plural = _("профили сотрудников")

    def __str__(self):
        return str(self.user_id)

    @cached_property
    def uid(self):
        return self.user.yauid

    def groups_str(self, start=None, end=None):
        if len(self.groups_tree) == 0:
            return ""

        names = self.get_group_names(
            self.groups_tree
        )

        if len(names) > 0:
            sliced_names = names
            if start is not None and end is not None:
                sliced_names = names[start:end]
            elif start is not None:
                sliced_names = names[start:]
            elif end is not None:
                sliced_names = names[:end]
            return " / ".join(sliced_names)

        return ""

    @cached_property
    def is_head(self):
        return self.leaderships.filter(role=StaffLeadership.ROLE_CHIEF, is_active=True).exists()

    @cached_property
    def is_deputy(self):
        return self.leaderships.filter(role=StaffLeadership.ROLE_DEPUTY, is_active=True).exists()

    @cached_property
    def leadership_joined_dates(self):
        return list(self.leaderships.filter(is_active=True).values_list('joined_at', flat=True))

    def mobile_phone_numbers(self):
        return [
            phone['number']
            for phone in self.phones or []
            if phone['type'] == 'mobile'
        ]

    def full_name(self, lang='ru'):
        """Return full name or username."""
        first_name = getattr(self, f'first_name_{lang}')
        last_name = getattr(self, f'last_name_{lang}')

        return " ".join(n for n in [first_name, last_name] if n)

    def first_name(self, lang='ru'):
        return getattr(self, f'first_name_{lang}')

    def last_name(self, lang='ru'):
        return getattr(self, f'last_name_{lang}')

    @property
    def full_name_ru(self):
        return self.full_name(lang='ru')

    @property
    def full_name_en(self):
        return self.full_name(lang='en')

    @property
    def staff_profile_url(self):
        return f'{settings.STAFF_BASE_URL}/{self.user.username}'

    def save(self, *args, **kwargs):
        if self.tracker.has_changed('office') and self.office:
            self.city = self.office.city

        super().save(*args, **kwargs)

    @staticmethod
    @cache_memoize(
        timeout=STAFF_PROFILE_GROUP_NAMES_TIMEOUT,
        args_rewrite=lambda pks: str(pks[-1]),
    )
    def get_group_names(pks: List[int]):
        qs = StaffGroup.objects.filter(pk__in=pks).values_list('name', flat=True)

        names = list(qs)
        names.reverse()

        return names


class StaffProfileForSupport(StaffProfile):
    class Meta:
        proxy = True
        verbose_name = _("[Для поддержки] Профиль на Стаффе")
        verbose_name_plural = _("[Для поддержки] Профили на Стаффе")


class StaffLeadership(TimeStampedModel):
    profile = models.ForeignKey(
        StaffProfile,
        verbose_name=_("профиль"),
        related_name='leaderships',
        on_delete=models.CASCADE,
    )
    group = models.IntegerField(
        verbose_name=_("подразделение"),
    )

    ROLE_CHIEF = 'chief'
    ROLE_DEPUTY = 'deputy'

    _ROLE_CHOICES = (
        (ROLE_CHIEF, _("руководитель")),
        (ROLE_DEPUTY, _("заместитель")),
    )
    role = models.CharField(_("роль"), max_length=255, choices=_ROLE_CHOICES, default=ROLE_CHIEF)
    joined_at = models.DateTimeField(null=True, blank=True)
    is_active = models.BooleanField(_("активно"), default=True)

    class Meta:
        unique_together = ('profile', 'group', 'role')
        verbose_name = _("руководство в подразделении")
        verbose_name_plural = _("руководство в подразделениях")
