from cache_memoize import cache_memoize

from django.contrib.auth import get_user_model
from django.db import models, transaction
from django.dispatch import receiver

from lms.contrib.staff.settings import STAFF_PROFILE_UPDATE_TIMEOUT
from lms.courses.services import flush_cache_course_available_for, flush_cache_courses_unavailable_for
from lms.users.signals import user_is_present

from .models import StaffProfile
from .services import profile_is_updated
from .tasks import get_staff_profile

User = get_user_model()


@cache_memoize(
    timeout=STAFF_PROFILE_UPDATE_TIMEOUT,
    store_result=False,
    args_rewrite=lambda user_pk, force_load=False: [user_pk],
)
def throttled_get_staff_profile(user_pk: int, force_load: bool = False):
    get_staff_profile.delay(user_pk, force_load)


@receiver(signal=models.signals.post_save, sender=User)
def user_post_save_handler(instance: User, created, **kwargs):
    if created or not getattr(instance, 'staffprofile', None):
        StaffProfile.objects.get_or_create(user=instance)
        transaction.on_commit(lambda: throttled_get_staff_profile(instance.pk, True))


@receiver(signal=user_is_present, dispatch_uid='update_staff_profile')
def user_logged_in_handler(user: User, **kwargs):
    if not profile_is_updated(user.pk):
        throttled_get_staff_profile(user.pk)


@receiver(signal=models.signals.post_save, sender=StaffProfile)
def staff_profile_post_save_handler(instance: StaffProfile, created, **kwargs):
    flush_cache_courses_unavailable_for(users_ids=[instance.user_id])
    flush_cache_course_available_for(users_ids=[instance.user_id])
